/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getValueProviderReferenceDescriptors;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import org.mule.weave.v2.parser.phase.CompilationException;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.valueprovider.ValueProviderDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class ValueProviderReferenceArgumentsHaveValidFormatRule extends DescriptorValidationRule {

  private static final String ERROR_TEMPLATE = "Value '%s' for argument '%s' does not match the expected format.";

  public ValueProviderReferenceArgumentsHaveValidFormatRule() {
    super("Value Provider arguments must respect the valid format (i.e. uriParameters.X, queryParameters.X, headers.X, "
        + "parameters.X, connection.X or configuration.X)",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> validationResults = new ArrayList<>();

    for (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor> reference : getValueProviderReferenceDescriptors(connectorDescriptor)) {
      final List<ResolverArgumentDescriptor> arguments = reference.getArguments();

      final List<ValidationResult> errors = arguments.stream()
          .filter(x -> !isValidArgumentValueFormat(x.getValue().getExpression()))
          .map(this::getValidationError)
          .collect(toList());

      validationResults.addAll(errors);
    }

    return validationResults;
  }

  private static String[] BINDINGS = new String[] {
      DWBindings.RESPONSE.getBinding(),
      DWBindings.REQUEST.getBinding(),
      ParameterType.URI.getBinding(),
      ParameterType.QUERY.getBinding(),
      ParameterType.HEADER.getBinding(),
      ParameterType.AUXILIAR.getBinding(),
      ParameterType.CONFIGURATION.getBinding(),
      ParameterType.CONNECTION.getBinding()
  };

  private boolean isValidArgumentValueFormat(String value) {
    if (isBlank(value)) {
      return false;
    }

    try {
      ExpressionHandlerUtils.compileDataWeaveScript(value, BINDINGS);
      return true;
    } catch (CompilationException e) {
      return false;
    }
  }

  private ValidationResult getValidationError(ArgumentDescriptor argument) {
    return new ValidationResult(this,
                                format(ERROR_TEMPLATE, argument.getValue().getExpression(), argument.getName()),
                                argument.getLocation());
  }
}
