/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.isWellFormedJSON;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.isWellFormedXML;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarBodyBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterRequestBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterResponseBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElement;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.apache.commons.lang3.tuple.Triple;

public class XmlAndJSONSchemasReferencedByDescriptorMustBeWellFormedRule extends DescriptorValidationRule {

  public XmlAndJSONSchemasReferencedByDescriptorMustBeWellFormedRule() {
    super("All XSD and JSON schemas referenced by the descriptor must be well formed.",
          "All Schema files referenced by the descriptor (as outputType, inputType or typeSchema) must be a well formed JSON or XML file.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> validationsResults = new ArrayList<>();
    List<Triple<String, String, DescriptorElement>> invalidSchemaReferences =
        getInvalidTypeSchemasFromDescriptor(connectorDescriptor);

    for (Triple<String, String, DescriptorElement> schemaReference : invalidSchemaReferences) {
      String detail = getDetailedElementDescription(schemaReference);
      DescriptorElementLocation location = schemaReference.getRight().getLocation();
      validationsResults.add(getValidationError(detail, location));
    }

    return validationsResults;
  }

  private ValidationResult getValidationError(String detail, DescriptorElementLocation location) {

    return new ValidationResult(this, detail, location);
  }

  private String getDetailedElementDescription(Triple<String, String, DescriptorElement> schemaReference) {

    StringBuilder mainElement = new StringBuilder();
    StringBuilder specificDescription = new StringBuilder();
    DescriptorElement descriptor = schemaReference.getRight();
    String mainElementName = schemaReference.getMiddle();
    String specificName = schemaReference.getLeft();


    if (descriptor instanceof TriggerDescriptor || descriptor instanceof TriggerParameterDescriptor) {
      mainElement.append("trigger '").append(mainElementName.toUpperCase()).append("'");

      if (descriptor instanceof TriggerParameterDescriptor) {
        specificDescription.append(", trigger parameter '").append(specificName.toUpperCase()).append("'");
      }
    } else {
      mainElement.append("operation '").append(mainElementName.toUpperCase()).append("'");

      if (descriptor instanceof OperationDescriptor) {
        specificDescription.append(", endpoint '").append(specificName.toUpperCase());
      } else {
        specificDescription.append(", property '").append(specificName.toUpperCase()).append("'");
      }
    }

    return String.format("The schema file declared in the %1$s%2$s is not a well formed JSON or XML file.", mainElement,
                         specificDescription);
  }

  private List<Triple<String, String, DescriptorElement>> getInvalidTypeSchemasFromDescriptor(ConnectorDescriptor connectorDescriptor) {

    ArrayList<Triple<String, String, DescriptorElement>> allInvalidSchemas = new ArrayList<>();
    allInvalidSchemas.addAll(getInvalidOperationAdaptersSchemasFromConnectorDescriptor(connectorDescriptor));
    allInvalidSchemas.addAll(getInvalidOperationSchemasFromConnectorDescriptor(connectorDescriptor));
    allInvalidSchemas.addAll(getInvalidTriggerSchemasFromConnectorDescriptor(connectorDescriptor));

    return allInvalidSchemas;
  }

  private List<Triple<String, String, DescriptorElement>> getInvalidTriggerSchemasFromConnectorDescriptor(ConnectorDescriptor connectorDescriptor) {

    List<Triple<String, String, DescriptorElement>> invalidTriggerSchemas = new ArrayList<>();
    List<TriggerDescriptor> triggers = getTriggersFromConnectorDescriptor(connectorDescriptor);

    for (TriggerDescriptor trigger : triggers) {
      String triggerName = trigger.getName();

      // Adds Trigger Schemas
      if (trigger.getOutputTypeSchema() != null && trigger.getOutputTypeSchema().getContent() != null
          && isInvalidSchema(trigger.getOutputTypeSchema().getContent())) {
        invalidTriggerSchemas.add(Triple.of(triggerName, triggerName, trigger));
      }

      // Adds Trigger Params Schemas
      trigger.getParameters()
          .stream()
          .filter(Objects::nonNull)
          .filter(x -> x.getInputType() != null && x.getInputType().getContent() != null
              && isInvalidSchema(x.getInputType().getContent()))
          .forEach(x -> invalidTriggerSchemas.add(Triple.of(x.getName(), triggerName, x)));
    }

    return invalidTriggerSchemas;
  }

  private List<Triple<String, String, DescriptorElement>> getInvalidOperationSchemasFromConnectorDescriptor(ConnectorDescriptor connectorDescriptor) {

    List<Triple<String, String, DescriptorElement>> invalidOperationSchemas = new ArrayList<>();

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      String endpointPath = endpoint.getPath().toUpperCase();
      for (OperationDescriptor operation : endpoint.getOperations()) {
        if (operation.getOutputTypeSchema() != null && operation.getOutputTypeSchema().getContent() != null
            && isInvalidSchema(operation.getOutputTypeSchema().getContent())) {
          invalidOperationSchemas.add(Triple.of(endpointPath + "' (OUTPUT)", operation.getMethod(), operation));
        }
        if (operation.getInputTypeSchema() != null && operation.getInputTypeSchema().getContent() != null
            && isInvalidSchema(operation.getInputTypeSchema().getContent())) {
          invalidOperationSchemas.add(Triple.of(endpointPath + "' (INPUT)", operation.getMethod(), operation));
        }
      }
    }

    return invalidOperationSchemas;
  }

  private List<Triple<String, String, DescriptorElement>> getInvalidOperationAdaptersSchemasFromConnectorDescriptor(ConnectorDescriptor connectorDescriptor) {

    List<Triple<String, String, DescriptorElement>> invalidOpAdaptersSchemas = new ArrayList<>();
    List<OperationAdapterDescriptor> operationAdapters = getOpAdaptersDescriptorStream(connectorDescriptor);

    for (OperationAdapterDescriptor operationAdapter : operationAdapters) {

      // Parameters Input Type
      String operationName = operationAdapter.getOperationId();
      for (AuxiliarParameterDescriptor parameter : operationAdapter.getParameters()) {
        if (parameter != null && parameter.getInputType() != null && parameter.getInputType().getContent() != null
            && isInvalidSchema(parameter.getInputType().getContent())) {
          invalidOpAdaptersSchemas.add(Triple.of(parameter.getName(), operationName, parameter));

        }
      }

      // Request Bindings Type Schemas
      AuxiliarParameterRequestBindingsDescriptor requestBindings = operationAdapter.getRequestBindings();
      if (requestBindings != null) {
        for (AuxiliarParameterBindingDescriptor parameterBinding : requestBindings.getQueryParameters()) {
          if (parameterBinding != null && parameterBinding.getInputType() != null
              && parameterBinding.getInputType().getContent() != null
              && isInvalidSchema(parameterBinding.getInputType().getContent())) {
            invalidOpAdaptersSchemas
                .add(Triple.of(parameterBinding.getName() + "request binding", operationName, parameterBinding));
          }
        }
      }

      // Response Bindings Type Schemas
      AuxiliarParameterResponseBindingsDescriptor responseBindings = operationAdapter.getResponseBindings();
      if (responseBindings != null && responseBindings.getBodyBinding() != null) {
        AuxiliarBodyBindingDescriptor bodyBinding = responseBindings.getBodyBinding();
        if (bodyBinding.getTypeSchema() != null && bodyBinding.getTypeSchema().getContent() != null
            && isInvalidSchema(bodyBinding.getTypeSchema().getContent())) {
          invalidOpAdaptersSchemas.add(Triple.of("response binding", operationName, bodyBinding));
        }
      }

    }
    return invalidOpAdaptersSchemas;
  }

  private boolean isInvalidSchema(String schemaString) {
    return (!isWellFormedJSON(schemaString) && !isWellFormedXML(schemaString));
  }

  private List<TriggerDescriptor> getTriggersFromConnectorDescriptor(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getTriggers().stream().filter(Objects::nonNull).collect(toList());
  }

  private List<OperationAdapterDescriptor> getOpAdaptersDescriptorStream(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getOperationAdapterDescriptors()
        .stream().filter(Objects::nonNull)
        .collect(toList());

  }
}
