/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.util;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.SchemaInferenceService;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.metadata.api.model.impl.DefaultStringType;
import org.mule.metadata.api.model.impl.DefaultNumberType;
import org.mule.metadata.api.model.impl.DefaultDateTimeType;
import org.mule.metadata.api.model.impl.DefaultDateType;
import org.mule.metadata.api.model.impl.DefaultLocalTimeType;
import org.mule.metadata.api.model.impl.DefaultTimeType;
import org.mule.metadata.api.model.impl.DefaultTimeZoneType;
import org.mule.metadata.api.model.impl.DefaultLocalDateTimeType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.api.model.impl.DefaultBooleanType;
import org.mule.metadata.api.model.impl.DefaultAnyType;
import org.mule.weave.v2.parser.phase.CompilationException;

import java.util.Optional;
import java.util.Map;
import java.util.Arrays;
import java.util.stream.Collectors;

import static com.mulesoft.connectivity.rest.commons.internal.metadatamodel.RestJsonTypeLoader.JSON;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class DataWeaveExpressionValidationUtils {

  public static Optional<ValidationResult> validateExpression(String expression, Map<String, MetadataType> bindings,
                                                              Class[] expectedMetadataTypes,
                                                              String identifier, ValidationRule rule) {

    if (expression != null && isExpressionSyntacticallyValid(expression, bindings.keySet().toArray(new String[0]))) {
      Optional<ValidationResult> validateExpressionMetadataTypeValidationResult =
          validateExpressionMetadataType(expression, bindings,
                                         expectedMetadataTypes,
                                         identifier, rule);
      return validateExpressionMetadataTypeValidationResult.isPresent() ? validateExpressionMetadataTypeValidationResult
          : Optional.empty();
    }
    return Optional.empty();
  }

  private static Optional<ValidationResult> validateExpressionMetadataType(String expression, Map<String, MetadataType> bindings,
                                                                           Class[] expectedMetadataTypes,
                                                                           String identifier, ValidationRule rule) {
    SchemaInferenceService schemaInferenceService = new SchemaInferenceService();
    MetadataType resultMetadataType = schemaInferenceService.inferSchema(expression, bindings);
    if (Arrays.stream(expectedMetadataTypes)
        .noneMatch(expectedMetadataType -> expectedMetadataType.equals(resultMetadataType.getClass()))) {
      return Optional.of(getValidationError(expression, identifier,
                                            format("Expected expression type is: %s but was %s",
                                                   String.join(" or ",
                                                               Arrays.stream(expectedMetadataTypes)
                                                                   .map(x -> getNameForMetadataTypeClass(x)).distinct()
                                                                   .collect(Collectors.toList())),
                                                   getNameForMetadataTypeClass(resultMetadataType.getClass())),
                                            rule));
    }
    return Optional.empty();
  }

  private static boolean isExpressionSyntacticallyValid(String expression,
                                                        String[] inputs) {
    if (isNotBlank(expression)) {
      try {
        ExpressionHandlerUtils.compileDataWeaveScript(expression, inputs);
      } catch (CompilationException e) {
        return false;
      }
    }
    return true;
  }

  public static MetadataType getMetadataTypeForTypeName(ParameterDataType typeName) {
    BaseTypeBuilder baseTypeBuilder = BaseTypeBuilder.create(JSON);
    switch (typeName) {
      case INTEGER:
      case NUMBER:
      case LONG:
        return baseTypeBuilder.numberType().build();
      case BOOLEAN:
      case BINARY:
        return baseTypeBuilder.booleanType().build();
      case LOCAL_DATE_TIME:
        return baseTypeBuilder.localDateTimeType().build();
      case ZONED_DATE_TIME:
        return baseTypeBuilder.timeZoneType().build();
      default:
        return baseTypeBuilder.stringType().build();
    }
  }

  private static String getNameForMetadataTypeClass(Class metadataTypeClassName) {
    if (DefaultArrayType.class.equals(metadataTypeClassName)) {
      return "Array";
    } else if (DefaultStringType.class.equals(metadataTypeClassName)) {
      return "String";
    } else if (DefaultNumberType.class.equals(metadataTypeClassName)) {
      return "Number";
    } else if (DefaultDateTimeType.class.equals(metadataTypeClassName) || DefaultDateType.class.equals(metadataTypeClassName)
        || DefaultLocalDateTimeType.class.equals(metadataTypeClassName) || DefaultLocalTimeType.class.equals(
                                                                                                             metadataTypeClassName)
        || DefaultTimeType.class.equals(metadataTypeClassName) || DefaultTimeZoneType.class.equals(
                                                                                                   metadataTypeClassName)) {
      return "Date";
    } else if (DefaultObjectType.class.equals(metadataTypeClassName)) {
      return "Object";
    } else if (DefaultBooleanType.class.equals(metadataTypeClassName)) {
      return "Boolean";
    } else if (DefaultAnyType.class.equals(metadataTypeClassName)) {
      return "Any";
    }
    return metadataTypeClassName.toString();
  }

  private static ValidationResult getValidationError(String expression,
                                                     String identifier, String exceptionMessage, ValidationRule rule) {
    String detail =
        format("DataWeave expression %s defined in %s does not match: \n %s", expression, identifier, exceptionMessage);

    return new ValidationResult(rule, detail);
  }

}
