/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.api;

import static com.mulesoft.connectivity.rest.sdk.internal.detection.util.SpecFormatDetectionUtils.detectSpecFormat;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.ApiParser.parseModel;
import static com.mulesoft.connectivity.rest.sdk.templating.TemplateEntity.getTemplateEngine;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.api.ApiConnectorModelLoader;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorConnectorModelLoader;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.InvalidSourceException;
import com.mulesoft.connectivity.rest.sdk.templating.api.ConnectorType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.exception.DescriptorParsingException;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParser;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationEngine;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;


public class RestSdk {

  private final File apiSpec;
  private final String rootDir;
  private final RestSdkRunConfiguration runConfiguration;
  private final List<File> descriptors = new ArrayList<>();
  private SpecFormat format;

  private ConnectorModel connectorModel;

  public RestSdk(File apiSpec, String rootDir, RestSdkRunConfiguration runConfiguration) {
    this.apiSpec = apiSpec;
    this.rootDir = rootDir;
    this.runConfiguration = runConfiguration;
  }

  public RestSdk format(SpecFormat format) {
    this.format = format;
    return this;
  }

  public RestSdk descriptor(File descriptor) {
    this.descriptors.add(descriptor);
    return this;
  }

  public ConnectorModel getConnectorModel() throws IOException, DescriptorParsingException, ModelGenerationException {

    SpecFormat specFormat = getFormat();

    List<ConnectorDescriptor> descriptorModels = getDescriptorModels();

    APIModel apiModel = parseModel(apiSpec, specFormat, rootDir, runConfiguration.skipValidations());

    ValidationEngine validationEngine = new ValidationEngine();

    // TODO: Simple Composition Validations
    boolean preValidationSuccess = validationEngine.preValidate(descriptorModels.get(0), apiModel);

    if (!preValidationSuccess) {
      throw new ModelGenerationException("There are (pre)validation errors.");
    }

    ConnectorModelBuilder connectorModelBuilder = ApiConnectorModelLoader.load(apiModel, new ConnectorModelBuilder());

    for (ConnectorDescriptor descriptorModel : descriptorModels) {
      DescriptorConnectorModelLoader.load(descriptorModel, connectorModelBuilder);
      // TODO: Simple Composition Validations. Validate each step?.
    }

    ConnectorModel connectorModel = connectorModelBuilder.build();

    // TODO: Simple Composition Validations. We might need to do this in the previous loop.
    boolean postValidationSuccess = validationEngine.postValidate(connectorModel);

    if (!postValidationSuccess) {
      throw new ModelGenerationException("There are (post)validation errors.");
    }

    this.connectorModel = connectorModel;
    return connectorModel;
  }

  public void generateConnector(ConnectorType connectorType, Path outputDir, RestSdkRunConfiguration runConfiguration)
      throws Exception {

    if (this.connectorModel == null) {
      getConnectorModel();
    }

    getTemplateEngine(connectorModel, descriptors, connectorType, runConfiguration, outputDir).applyTemplates();
  }

  private SpecFormat getFormat() throws InvalidSourceException, IOException {
    if (this.format != null) {
      return this.format;
    }

    return detectSpecFormat(apiSpec);
  }

  private List<ConnectorDescriptor> getDescriptorModels() throws DescriptorParsingException {

    List<ConnectorDescriptor> descriptorModels = new ArrayList<>();
    DescriptorParser descriptorParser = new DescriptorParser();

    for (File descriptorFile : descriptors) {
      descriptorModels.add(descriptorParser.parseConnectorDescriptor(descriptorFile));
    }

    return descriptorModels;
  }

}
