/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.INFO;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.WARN;
import static java.lang.System.lineSeparator;
import static java.util.stream.Collectors.toList;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;

import org.slf4j.Logger;

public class ValidationEngine {

  private static final Logger LOGGER = getLogger(ValidationEngine.class);
  private static final ModelValidator VALIDATOR = new ModelValidator();

  private ConnectorDescriptor connectorDescriptor;
  private APIModel apiModel;

  public boolean preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    this.connectorDescriptor = connectorDescriptor;
    this.apiModel = apiModel;

    List<ValidationError> validationErrors = VALIDATOR.preValidate(apiModel, connectorDescriptor);
    reportValidationResults(validationErrors);

    return validationErrors.stream().noneMatch(x -> x.getLevel().equals(ERROR));
  }

  public boolean postValidate(ConnectorModel connectorModel) {

    List<ValidationError> validationErrors = VALIDATOR.postValidate(apiModel, connectorDescriptor, connectorModel);
    reportValidationResults(validationErrors);

    return validationErrors.stream().noneMatch(x -> x.getLevel().equals(ERROR));
  }

  private void reportValidationResults(List<ValidationError> validationErrors) {
    List<ValidationError> errorList = validationErrors.stream()
        .filter(x -> x.getLevel().equals(ERROR))
        .collect(toList());

    List<ValidationError> warnList = validationErrors.stream()
        .filter(x -> x.getLevel().equals(WARN))
        .collect(toList());

    List<ValidationError> infoList = validationErrors.stream()
        .filter(x -> x.getLevel().equals(INFO))
        .collect(toList());

    errorList.forEach(x -> LOGGER.error(x.toString() + lineSeparator()));
    warnList.forEach(x -> LOGGER.warn(x.toString() + lineSeparator()));
    infoList.forEach(x -> LOGGER.warn(x.toString() + lineSeparator()));
  }
}
