/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class DescriptorAdditionalParameterMandatoryFieldsRule extends PreValidationRule {

  public DescriptorAdditionalParameterMandatoryFieldsRule() {
    super("Additional security parameters declared in the connector descriptor must have mandatory fields",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationError> results = new ArrayList<>();

    for (SecuritySchemeDescriptor securitySchemeDescriptor : connectorDescriptor.getSecurity()) {

      if (securitySchemeDescriptor.getQueryParameters() != null) {
        validateParametersExists(securitySchemeDescriptor.getQueryParameters(),
                                 securitySchemeDescriptor.getName(), results);
      }

      if (securitySchemeDescriptor.getHeaders() != null) {
        validateParametersExists(securitySchemeDescriptor.getHeaders(),
                                 securitySchemeDescriptor.getName(), results);
      }
    }

    return results;
  }

  private void validateParametersExists(List<ParameterDescriptor> parameters,
                                        String securityScheme,
                                        List<ValidationError> validationErrors) {
    parameters.stream()
        .filter(ParameterDescriptor::isAdditional)
        .forEach(paramDesc -> validateParameterFields(paramDesc, securityScheme, validationErrors));
  }

  private void validateParameterFields(ParameterDescriptor parameter,
                                       String securityScheme,
                                       List<ValidationError> validationErrors) {
    if (parameter.getDataType() == null) {
      validationErrors.add(getValidationError(parameter, securityScheme, "type"));
    }
    if (isBlank(parameter.getDisplayName())) {
      validationErrors.add(getValidationError(parameter, securityScheme, "displayName"));
    }
  }

  private ValidationError getValidationError(ParameterDescriptor paramDescriptor, String securityScheme, String field) {
    String detail = format("Field '%s' from parameter '%s' in Security scheme '%s' is missing", field,
                           paramDescriptor.getParamName(), securityScheme);

    return new ValidationError(this, detail, paramDescriptor.getLocation());
  }
}
