/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class DescriptorAdditionalParameterMustNotBePresentInApiRule extends PreValidationRule {

  public DescriptorAdditionalParameterMustNotBePresentInApiRule() {
    super("Operation additional parameters declared in the connector descriptor must not be present in the API spec",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationError> results = new ArrayList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        if (operationDescriptor.getExpects() != null) {
          results.addAll(validateParametersExists(operationDescriptor.getExpects().getUriParameter(), URI,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getQueryParameter(), QUERY,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getHeader(), HEADER,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

        }
      }
    }

    return results;
  }

  private List<ValidationError> validateParametersExists(List<ParameterDescriptor> parameters,
                                                         ParameterType parameterType,
                                                         String path,
                                                         String method,
                                                         APIModel apiModel) {
    return parameters.stream()
        .filter(ParameterDescriptor::isAdditional)
        .map(paramDesc -> validateParameterNotExists(paramDesc, parameterType, path, method, apiModel))
        .filter(Objects::nonNull)
        .collect(toList());
  }

  private ValidationError validateParameterNotExists(ParameterDescriptor parameter,
                                                     ParameterType parameterType,
                                                     String path,
                                                     String method,
                                                     APIModel apiModel) {

    return apiModel.getOperationsModel().stream()
        .filter(apiOp -> apiOp.getPath().equalsIgnoreCase(path))
        .filter(apiOp -> apiOp.getHttpMethod().equalsIgnoreCase(method))
        .findFirst()
        .map(x -> validateParameterExists(x, parameter, parameterType))
        .orElse(null);
  }

  private ValidationError validateParameterExists(APIOperationModel apiOperation, ParameterDescriptor parameter,
                                                  ParameterType parameterType) {
    List<APIParameterModel> apiParameterList;

    if (parameterType.equals(URI)) {
      apiParameterList = apiOperation.getUriParamsModel();
    } else if (parameterType.equals(QUERY)) {
      apiParameterList = apiOperation.getQueryParamsModel();
    } else {
      apiParameterList = apiOperation.getHeadersModel();
    }

    return apiParameterList.stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameter.getParamName()))
        ? getValidationError(apiOperation, parameter, parameterType)
        : null;
  }

  private ValidationError getValidationError(APIOperationModel apiOperationModel, ParameterDescriptor paramDescriptor,
                                             ParameterType parameterType) {
    String detail = format("Descriptor declares additional %s '%s' that exists in the API spec for path '%s', method '%s'",
                           parameterType.getName(), paramDescriptor.getParamName(), apiOperationModel.getPath(),
                           apiOperationModel.getHttpMethod().toUpperCase());

    return new ValidationError(this, detail, paramDescriptor.getLocation());
  }
}
