/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationError;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;

public class NoOutputMediaTypeDefinedRule extends PreValidationRule {

  // R019
  public NoOutputMediaTypeDefinedRule() {
    super("Defining a media type for the operations output type is mandatory",
          "If an operation defines an output type it should also define a media type. This can be done at the operation level or globally.",
          ERROR);
  }

  @Override
  public List<ValidationError> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    if (isNotBlank(connectorDescriptor.getDefaultOutputMediaType())) {
      return emptyList();
    }

    List<ValidationError> allErrors = new ArrayList<>();

    apiModel.getOperationsModel().stream()
        .filter(apiOp -> !apiOp.getOutputMetadataModel().isEmpty())
        .filter(apiOp -> !outputMediaTypeDefinedInAPISpec(apiOp))
        .filter(apiOp -> !outputMediaTypeDefinedInDescriptor(apiOp, connectorDescriptor))
        .map(apiOp -> getValidationErrorForApiSpec(apiOp, connectorDescriptor))
        .forEach(allErrors::add);

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      endpoint.getOperations().stream()
          .filter(opDesc -> isNotBlank(opDesc.getOutputTypeSchema()))
          .filter(opDesc -> isBlank(opDesc.getOutputMediaType()))
          .filter(opDesc -> !outputMediaTypeDefinedInAPISpec(apiModel, endpoint.getPath(), opDesc.getMethod()))
          .map(this::getValidationErrorForDescriptor)
          .forEach(allErrors::add);
    }

    return allErrors;
  }

  private boolean outputMediaTypeDefinedInAPISpec(APIOperationModel apiOperation) {
    return apiOperation.getOutputMetadataModel().stream()
        .anyMatch(x -> x.getMediaType() != null);
  }

  private boolean outputMediaTypeDefinedInAPISpec(APIModel apiModel, String path, String method) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().equalsIgnoreCase(method))
        .flatMap(x -> x.getOutputMetadataModel().stream())
        .anyMatch(x -> x.getMediaType() != null);
  }

  private boolean outputMediaTypeDefinedInDescriptor(APIOperationModel apiOperationModel, ConnectorDescriptor descriptor) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(descriptor, apiOperationModel);
    return operationDescriptor != null && isNotBlank(operationDescriptor.getOutputMediaType());
  }

  private ValidationError getValidationErrorForApiSpec(APIOperationModel apiOperationModel,
                                                       ConnectorDescriptor connectorDescriptor) {
    String detail =
        "API Operation with PATH: "
            + apiOperationModel.getPath()
            + " and METHOD: "
            + apiOperationModel.getHttpMethod()
            + " defined in the API spec does not declare a media type for its output type";

    return new ValidationError(this, EMPTY, connectorDescriptor.getLocation());
  }

  private ValidationError getValidationErrorForDescriptor(OperationDescriptor operationDescriptor) {
    return new ValidationError(this, EMPTY, operationDescriptor.getLocation());
  }
}
