/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class NoOutputMediaTypeDefinedRule extends ConnectorModelBuilderValidationRule {

  // R019
  public NoOutputMediaTypeDefinedRule() {
    super("Defining a media type for the operations output type is mandatory",
          "If an operation defines an output type it should also define a media type. This can be done at the operation level or globally.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(ConnectorModelBuilder connectorModelBuilder) {
    if (connectorModelBuilder.getDefaultOutputMediaType() != null) {
      return new ArrayList<>();
    }

    return connectorModelBuilder.getOperationBuilders().stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(this::validateOperation)
        .filter(Objects::nonNull)
        .collect(toList());
  }

  private ValidationResult validateOperation(OperationBuilder operationBuilder) {
    if (operationBuilder.getDefaultOutputMediaType() != null) {
      return null;
    }

    if ((!operationBuilder.getOutputMetadataBuilders().isEmpty() || operationBuilder.hasForcedOutputTypeSchema()) &&
        operationBuilder.getOutputMetadataBuilders().keySet().stream().allMatch(Objects::isNull)) {
      return getValidationErrorForApiSpec(operationBuilder);
    }

    return null;
  }

  private ValidationResult getValidationErrorForApiSpec(OperationBuilder operationBuilder) {
    String detail =
        "Operation with PATH: "
            + operationBuilder.getPath()
            + " and METHOD: "
            + operationBuilder.getMethod().toUpperCase()
            + " does not declare a media type for its output type.";

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
