/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.GET;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.HttpMethodDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataSameOperationDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class SampleDataSameOperationMustBeGETRule extends DescriptorValidationRule {

  public SampleDataSameOperationMustBeGETRule() {
    super("Sample Data Same Operation cannot be used in POST/PUT/DELETE/PATCH methods, ignoring this one.",
          "Sample Data Same Operation is only available for GET method, others will be ignored.",
          WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> results = new ArrayList<>();

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operation : endpoint.getOperations()) {
        String method = operation.getMethod();
        if (invalidMethodForSameOperation(operation.getSampleDataExpressionDescriptor(), method)) {
          ValidationResult vr = new ValidationResult(this, EMPTY, operation.getLocation());
          results.add(vr);
        }
      }
    }

    for (TriggerDescriptor trigger : connectorDescriptor.getTriggers()) {
      HttpMethodDescriptor method = trigger.getMethod();
      if (invalidMethodForSameOperation(trigger.getSampleDataExpressionDescriptor(), method.getName())) {
        ValidationResult vr = new ValidationResult(this, EMPTY, trigger.getLocation());
        results.add(vr);
      }
    }

    return results;
  }

  private boolean invalidMethodForSameOperation(ResolverExpressionDescriptor<SampleDataDefinitionDescriptor> resolverExpression,
                                                String method) {

    if (resolverExpression instanceof ResolverReferenceDescriptor) {
      resolverExpression =
          ((ResolverReferenceDescriptor<SampleDataDefinitionDescriptor>) resolverExpression).getDeclaration().getDefinition();
    }

    if (resolverExpression instanceof SampleDataSameOperationDefinitionDescriptor) {
      return HTTPMethod.fromString(method.toUpperCase()) != GET;
    }

    return false;
  }
}
