/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.INFO;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.lang.System.lineSeparator;
import static java.util.stream.Collectors.toList;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ApiValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.ConnectorMustHaveCompleteGavRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.TriggerMandatoryFieldsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.BaseJavaPackageMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ConnectorNameMustBeFriendlyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorAdditionalParameterMandatoryFieldsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorRedundantOutputDefinitionRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ExtensionXmlMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.GavArtifactIdMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.GavGroupIdMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.GavVersionMustBeValidRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationDisplayNameScriptMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.OperationIdentifierScriptMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.PagedOperationsExpressionMustCompileRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.PagedOperationsParameterMustExistRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.SampleDataSameOperationMustBeGETRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceArgumentsExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceArgumentsHaveValidFormatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorAdditionalParameterMustNotBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorInputMediaTypeMustBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorOperationMustExistInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorOtputMediaTypeMustBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorParameterMustBePresentInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.DescriptorPathMustExistInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerBindingParameterMustExistInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerPathAndMethodMustExistInApiRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.TriggerUriParamsMustBeCompleteRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.NoDefaultInputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.NoDefaultOutputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.NoInputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.NoOutputMediaTypeDefinedRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.OperationIdentifierScriptMustNotRepeatRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.OperationMustDeclareResponseBodyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.OperationNameMustBeFriendlyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.PagedOperationsOperationMustDeclareResponseBodyRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.api.AtLeastOneSupportedSecuritySchemeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.api.NotSupportedSecuritySchemeRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.ValueProviderReferenceArgumentsOperationParameterExistsRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;

public class ValidationEngine {

  private static final Logger LOGGER = getLogger(ValidationEngine.class);

  private final List<ValidationResult> validationResults = new ArrayList<>();
  private final List<ValidationRule> rules;

  public ValidationEngine() {
    this.rules = ALL_RULES;
  }

  public ValidationEngine(List<ValidationRule> rules) {
    this.rules = rules;
  }

  public void validateApi(APIModel apiModel) {
    List<ValidationResult> apiResults = rules.stream()
        .filter(x -> x instanceof ApiValidationRule)
        .flatMap(x -> ((ApiValidationRule) x).validate(apiModel).stream())
        .collect(toList());

    validationResults.addAll(apiResults);
  }

  public void validateDescriptor(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> descriptorResults = rules.stream()
        .filter(x -> x instanceof DescriptorValidationRule)
        .flatMap(x -> ((DescriptorValidationRule) x).validate(apiModel, connectorDescriptor).stream())
        .collect(toList());

    validationResults.addAll(descriptorResults);
  }

  public void validateConnectorModelBuilder(ConnectorModelBuilder connectorModelBuilder) {
    List<ValidationResult> connectorResults = rules.stream()
        .filter(x -> x instanceof ConnectorModelBuilderValidationRule)
        .flatMap(x -> ((ConnectorModelBuilderValidationRule) x).validate(connectorModelBuilder).stream())
        .collect(toList());

    validationResults.addAll(connectorResults);
  }

  public void validateConnectorModel(ConnectorModel connectorModel) {
    List<ValidationResult> connectorResults = rules.stream()
        .filter(x -> x instanceof ConnectorModelValidationRule)
        .flatMap(x -> ((ConnectorModelValidationRule) x).validate(connectorModel).stream())
        .collect(toList());

    validationResults.addAll(connectorResults);
  }

  public boolean isSuccess() {
    return validationResults.stream().noneMatch(x -> x.getLevel().equals(ERROR));
  }

  public void logResults() {
    validationResults.stream()
        .filter(x -> x.getLevel().equals(ERROR))
        .forEach(x -> LOGGER.error(x + lineSeparator()));

    validationResults.stream()
        .filter(x -> x.getLevel().equals(WARN))
        .forEach(x -> LOGGER.warn(x + lineSeparator()));

    validationResults.stream()
        .filter(x -> x.getLevel().equals(INFO))
        .forEach(x -> LOGGER.warn(x + lineSeparator()));
  }

  public List<ValidationResult> getResults() {
    return validationResults;
  }

  public static final List<ValidationRule> ALL_RULES = new ArrayList<>();
  static {
    ALL_RULES.add(new DescriptorPathMustExistInApiRule());
    ALL_RULES.add(new DescriptorOperationMustExistInApiRule());
    ALL_RULES.add(new OperationMustDeclareResponseBodyRule());
    ALL_RULES.add(new AtLeastOneSupportedSecuritySchemeRule());
    ALL_RULES.add(new ExtensionXmlMustBeValidRule());
    ALL_RULES.add(new BaseJavaPackageMustBeValidRule());
    ALL_RULES.add(new GavGroupIdMustBeValidRule());
    ALL_RULES.add(new GavArtifactIdMustBeValidRule());
    ALL_RULES.add(new GavVersionMustBeValidRule());
    ALL_RULES.add(new NotSupportedSecuritySchemeRule());
    ALL_RULES.add(new ConnectorNameMustBeFriendlyRule());
    ALL_RULES.add(new OperationNameMustBeFriendlyRule());
    ALL_RULES.add(new OperationDisplayNameScriptMustCompileRule());
    ALL_RULES.add(new OperationIdentifierScriptMustCompileRule());
    ALL_RULES.add(new OperationIdentifierScriptMustNotRepeatRule());
    ALL_RULES.add(new NoDefaultInputMediaTypeDefinedRule());
    ALL_RULES.add(new NoDefaultOutputMediaTypeDefinedRule());
    ALL_RULES.add(new DescriptorParameterMustBePresentInApiRule());
    ALL_RULES.add(new NoInputMediaTypeDefinedRule());
    ALL_RULES.add(new DescriptorInputMediaTypeMustBePresentInApiRule());
    ALL_RULES.add(new NoOutputMediaTypeDefinedRule());
    ALL_RULES.add(new DescriptorOtputMediaTypeMustBePresentInApiRule());
    ALL_RULES.add(new DescriptorRedundantOutputDefinitionRule());
    ALL_RULES.add(new PagedOperationsOperationMustDeclareResponseBodyRule());
    ALL_RULES.add(new TriggerPathAndMethodMustExistInApiRule());
    ALL_RULES.add(new TriggerBindingParameterMustExistInApiRule());
    ALL_RULES.add(new TriggerUriParamsMustBeCompleteRule());
    ALL_RULES.add(new ValueProviderReferenceArgumentsExistsRule());
    ALL_RULES.add(new ValueProviderReferenceArgumentsHaveValidFormatRule());
    ALL_RULES.add(new SampleDataSameOperationMustBeGETRule());
    ALL_RULES.add(new PagedOperationsParameterMustExistRule());
    ALL_RULES.add(new PagedOperationsExpressionMustCompileRule());
    ALL_RULES.add(new DescriptorAdditionalParameterMandatoryFieldsRule());
    ALL_RULES.add(new DescriptorAdditionalParameterMustNotBePresentInApiRule());
    ALL_RULES.add(new ConnectorMustHaveCompleteGavRule());
    ALL_RULES.add(new ValueProviderReferenceArgumentsOperationParameterExistsRule());
    ALL_RULES.add(new TriggerMandatoryFieldsRule());
  }
}
