/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.PaginationType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;

public class PagedOperationsParameterMustExistRule extends DescriptorValidationRule {

  public PagedOperationsParameterMustExistRule() {
    super("Paged operations must contain a query parameter corresponding to the pagination parameter declared in the pagination definition",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();

    for (APIOperationModel apiOperation : apiModel.getOperationsModel()) {
      PaginationDeclarationDescriptor paginationDeclaration = getPaginationDeclaration(connectorDescriptor, apiOperation);
      if (paginationDeclaration != null
          && !PaginationType.HYPERMEDIA.getName().equalsIgnoreCase(paginationDeclaration.getType())) {
        String parameterName = paginationDeclaration.getPaginationParameterName();
        if (apiOperation.getQueryParamsModel().stream().noneMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName))) {
          validationResults
              .add(getValidationError(connectorDescriptor, apiOperation, parameterName, paginationDeclaration.getName()));
        }
      }
    }

    return validationResults;
  }

  private PaginationDeclarationDescriptor getPaginationDeclaration(ConnectorDescriptor connectorDescriptor,
                                                                   APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);

    PaginationDeclarationDescriptor paginationDescriptor = null;
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())) {
      paginationDescriptor = connectorDescriptor.getPaginations().stream()
          .filter(x -> x.getName().equalsIgnoreCase(operationDescriptor.getPagination()))
          .findFirst().orElse(null);
    }

    return paginationDescriptor;
  }

  private ValidationResult getValidationError(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel,
                                              String parameterName, String paginationName) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);
    String description = "The pagination named '" + paginationName + "' declares a query parameter named '" + parameterName
        + "' that is not present in the API spec for the referencing operation";

    return new ValidationResult(this, description, operationDescriptor.getLocation());
  }
}
