/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.INFO;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TypeDefinitionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.CompleteConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.ws.rs.core.MediaType;

public class NoDefaultInputMediaTypeDefinedRule extends CompleteConnectorModelBuilderValidationRule {

  // R014
  public NoDefaultInputMediaTypeDefinedRule() {
    super("No default inputMediaType was set in the Descriptor and multiple are available",
          EMPTY,
          INFO);
  }

  @Override
  public List<ValidationResult> validate(ConnectorModelBuilder connectorModelBuilder) {
    return connectorModelBuilder.getOperationBuilders().stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(opBuilder -> validateOperation(connectorModelBuilder, opBuilder))
        .filter(Objects::nonNull)
        .collect(toList());
  }

  public ValidationResult validateOperation(ConnectorModelBuilder connectorModelBuilder,
                                            OperationBuilder operationBuilder) {
    Map<MediaType, TypeDefinitionBuilder> inputMetadataBuilders = operationBuilder.getInputMetadataBuilders();

    if (inputMetadataBuilders.size() > 1) {
      MediaType defaultInputMediaType = defaultIfNull(operationBuilder.getDefaultInputMediaType(),
                                                      connectorModelBuilder.getDefaultInputMediaType());

      TypeDefinitionBuilder builder = inputMetadataBuilders.get(defaultInputMediaType);

      if (builder == null) {
        return getValidationError(operationBuilder);
      }
    }

    return null;
  }

  private ValidationResult getValidationError(OperationBuilder operationBuilder) {
    final String detail =
        "API Operation with PATH: "
            + operationBuilder.getPath()
            + " and METHOD: "
            + operationBuilder.getMethod().toUpperCase()
            + " declares multiple input media types but none was selected in the Operation descriptor.";

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
