/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getValueProviderReferenceDescriptors;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.valueprovider.ValueProviderDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class ValueProviderReferenceArgumentsExistsRule extends DescriptorValidationRule {

  private static final String ERROR_TEMPLATE = "Argument '%s' does not exist in the '%s' value provider declaration.";

  public ValueProviderReferenceArgumentsExistsRule() {
    super("When sending an argument to a value provider it must exist in the resolver declaration", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> validationResults = new ArrayList<>();

    for (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor> reference : getValueProviderReferenceDescriptors(connectorDescriptor)) {
      final List<ResolverArgumentDescriptor> arguments = reference.getArguments();
      final List<ResolverParameterDescriptor> parameters = reference.getDeclaration().getParameters();

      final List<ValidationResult> errors = arguments.stream()
          .filter(x -> parameters.stream().noneMatch(y -> y.getName().equalsIgnoreCase(x.getName())))
          .map(x -> getValidationError(x, reference.getDeclaration()))
          .collect(toList());

      validationResults.addAll(errors);
    }

    return validationResults;
  }

  private ValidationResult getValidationError(ArgumentDescriptor argumentDescriptor,
                                              ResolverDeclarationDescriptor<?> declaration) {
    return new ValidationResult(this,
                                format(ERROR_TEMPLATE, argumentDescriptor.getName(), declaration.getName()),
                                argumentDescriptor.getLocation());
  }
}
