/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;

public class DescriptorOperationMustExistInApiRule extends DescriptorValidationRule {

  // R002
  public DescriptorOperationMustExistInApiRule() {
    super("Descriptor operations must be present in the API spec",
          "Any operation (PATH + METHOD) declared in the connector descriptor must be present in the API spec.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<Pair<String, OperationDescriptor>> descriptorOperations =
        connectorDescriptor.getEndpoints().stream()
            .flatMap(endpointDesc -> endpointDesc.getOperations().stream().map(opDesc -> Pair.of(endpointDesc.getPath(), opDesc)))
            .collect(toList());

    List<Pair<String, String>> apiOperations = apiModel.getOperationsModel().stream()
        .map(x -> Pair.of(x.getPath(), x.getHttpMethod()))
        .distinct()
        .collect(toList());

    List<ValidationResult> validationResults = new ArrayList<>();

    for (Pair<String, OperationDescriptor> descriptorOperation : descriptorOperations) {
      if (apiOperations.stream().noneMatch(
                                           apiOp -> apiOp.getKey().equalsIgnoreCase(descriptorOperation.getKey())
                                               && apiOp.getValue()
                                                   .equalsIgnoreCase(descriptorOperation.getValue().getMethod()))) {

        validationResults.add(getValidationError(descriptorOperation));
      }
    }

    return validationResults;
  }

  private ValidationResult getValidationError(Pair<String, OperationDescriptor> descriptorOperation) {

    String detail = "No operation with PATH: "
        + descriptorOperation.getKey()
        + " and METHOD: "
        + descriptorOperation.getValue().getMethod().toUpperCase()
        + " exists in the API spec";

    return new ValidationResult(this, detail, descriptorOperation.getValue().getLocation());
  }
}
