/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationIdentifierExpressionHandler.IMPLICIT_INPUTS;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.joining;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class OperationIdentifierScriptMustNotRepeatRule extends ConnectorModelValidationRule {

  public OperationIdentifierScriptMustNotRepeatRule() {
    super("Operation identifier expression must not repeat values among operations",
          format("The 'operationIdentifier' must be revisited (available bindings are '%s')", Arrays.toString(IMPLICIT_INPUTS)),
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(ConnectorModel connectorModel) {
    Map<String, List<ConnectorOperation>> operationIdentifiers = new HashMap<>();

    for (ConnectorOperation op : connectorModel.getOperations()) {
      if (!operationIdentifiers.containsKey(op.getOperationIdentifier())) {
        operationIdentifiers.put(op.getOperationIdentifier(), new ArrayList<>());
      }
      operationIdentifiers.get(op.getOperationIdentifier()).add(op);
    }

    List<ConnectorOperation> duplicatedOperations = new ArrayList<>();
    operationIdentifiers.forEach((id, ops) -> {
      if (ops.size() > 1) {
        duplicatedOperations.addAll(ops);
      }
    });

    if (duplicatedOperations.isEmpty()) {
      return emptyList();
    }

    return singletonList(getValidationError(duplicatedOperations));
  }

  private ValidationResult getValidationError(List<ConnectorOperation> operations) {

    String repeatedIds = operations.stream()
        .map(op -> op.getHttpMethod() + ":" + op.getPath() + " -> " + op.getOperationIdentifier())
        .collect(joining(";\n"));

    String detail = format("The generated 'operationIdentifier' is repeated for some operations. Please revisit the script.\n%s",
                           repeatedIds);

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
