/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.util.stream.Collectors.joining;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class OperationInternalNameMustNotRepeatRule extends ConnectorModelValidationRule {

  public OperationInternalNameMustNotRepeatRule() {
    super("Operation's internalName must not be repeated among operations.",
          "The internal name in some operations, used for java classes generation and generated from 'operationIdentifier' script, is colliding. "
              +
              "Please re-arrange the 'operationIdentifier' script accordingly to avoid collisions",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(ConnectorModel connectorModel) {
    Map<String, List<ConnectorOperation>> internalNames = new HashMap<>();

    for (ConnectorOperation op : connectorModel.getOperations()) {
      if (!internalNames.containsKey(op.getInternalName())) {
        internalNames.put(op.getInternalName(), new ArrayList<>());
      }
      internalNames.get(op.getInternalName()).add(op);
    }
    List<ValidationResult> validationResults = internalNames.entrySet().stream()
        .filter(entry -> entry.getValue().size() > 1)
        .map(entry -> getValidationError(entry.getKey(), entry.getValue()))
        .collect(Collectors.toList());
    return validationResults;
  }

  private ValidationResult getValidationError(String internalName, List<ConnectorOperation> operations) {

    String repeatedInternalNameOps = operations.stream()
        .map(op -> format("%s:%s with operation identifier [%s]", op.getHttpMethod(), op.getPath(), op.getOperationIdentifier()))
        .collect(joining("\n"));

    String detail =
        format("The following operations generated the same internal name [%s] from the operationIdentifier, please revisit the script 'operationIdentifier'"
            +
            ".\n%s",
               internalName,
               repeatedInternalNameOps);

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
