/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperation;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.apache.commons.lang3.StringUtils.isEmpty;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;

public class TriggerPathAndMethodMustExistInApiRule extends DescriptorValidationRule {

  public TriggerPathAndMethodMustExistInApiRule() {
    super("Trigger PATH and METHOD must exist in the API specification.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (triggerDescriptors != null && !triggerDescriptors.isEmpty()) {
      return triggerDescriptors.stream()
          .filter(x -> !hasMatchingOperation(apiModel, x))
          .map(this::getValidationError)
          .collect(toList());
    }

    return emptyList();
  }

  private static boolean hasMatchingOperation(APIModel apiModel, TriggerDescriptor triggerDescriptor) {
    if (isEmpty(triggerDescriptor.getPath()) || triggerDescriptor.getMethod() == null) {
      return true;
    }

    return getApiOperation(apiModel, triggerDescriptor) != null;
  }

  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor) {
    String detail =
        "Trigger with PATH: '"
            + triggerDescriptor.getPath()
            + "' and METHOD: '"
            + triggerDescriptor.getMethod().getName().toUpperCase()
            + "' does not exist in the API spec";

    return new ValidationResult(this, detail, triggerDescriptor.getLocation());
  }
}
