/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete;

import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.CompleteConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.ws.rs.core.MediaType;

public class NoInputMediaTypeDefinedRule extends CompleteConnectorModelBuilderValidationRule {

  // R017
  public NoInputMediaTypeDefinedRule() {
    super("Defining a media type for the operations input type is mandatory",
          "If an operation defines an input type it should also define a media type. This can be done at the operation level or globally.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(ConnectorModelBuilder connectorModelBuilder) {
    if (connectorModelBuilder.getDefaultInputMediaType() != null) {
      return new ArrayList<>();
    }

    return connectorModelBuilder.getOperationBuilders().stream()
        .filter(op -> op.isIgnored() == null || !op.isIgnored())
        .filter(op -> !op.isAdapter()) // TODO: This validations does not apply for adapters. We need to build a similar one.
        .map(op -> this.validateOperation(op, connectorModelBuilder.getDefaultInputMediaType()))
        .filter(Objects::nonNull)
        .collect(toList());
  }

  private ValidationResult validateOperation(OperationBuilder operationBuilder, MediaType globalMediaType) {
    if (operationBuilder.getDefaultInputMediaType() != null) {
      return null;
    }

    if ((!operationBuilder.getInputMetadataBuilders().isEmpty() || operationBuilder.hasForcedInputTypeSchema()) &&
        operationBuilder.getInputMetadataBuilders().keySet().stream().allMatch(Objects::isNull)) {

      if (globalMediaType == null && operationBuilder.getDefaultInputMediaType() == null) {
        return getValidationErrorForApiSpec(operationBuilder);
      }
    }

    return null;
  }

  private ValidationResult getValidationErrorForApiSpec(OperationBuilder operationBuilder) {
    final String detail =
        "Operation with PATH: "
            + operationBuilder.getPath()
            + " and METHOD: "
            + operationBuilder.getMethod().toUpperCase()
            + " does not declare a media type for its input type.";

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
