/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.BODY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.util.Collections.emptyList;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.join;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterRequestBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

//TODO RSDK-748: this validation is not currently working for scenarios like parameters.&'someParamName', parameters['someParamName']
public class ParametersUsedInOperationAdapterBindingsShouldExistRule extends DescriptorValidationRule {

  private static final Pattern PARAMETER_EXTRACTION_PATTERN = Pattern.compile("parameters\\.(.*?)[^a-zA-Z0-9]");

  public ParametersUsedInOperationAdapterBindingsShouldExistRule() {
    super("Parameter used in binding expression should exist.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    List<OperationAdapterDescriptor> operationAdapterDescriptors = connectorDescriptor.getOperationAdapterDescriptors();
    for (OperationAdapterDescriptor operationAdapterDescriptor : operationAdapterDescriptors) {
      AuxiliarParameterRequestBindingsDescriptor requestBindings = operationAdapterDescriptor.getRequestBindings();
      if (requestBindings != null) {
        List<String> parametersAvailable = operationAdapterDescriptor.getParameters().stream()
            .map(AuxiliarParameterDescriptor::getName).collect(Collectors.toList());
        analyzeParameters(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getHeaders(), HEADER);
        analyzeParameters(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getQueryParameters(), QUERY);
        analyzeParameters(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getUriParameters(), URI);
        analyzeBody(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getRequestBodyExpression());
      }
    }

    return results;
  }

  private void analyzeParameters(List<ValidationResult> results, OperationAdapterDescriptor operationAdapterDescriptor,
                                 List<String> parametersAvailable, List<AuxiliarParameterBindingDescriptor> bindings,
                                 ParameterType bindingParameterType) {

    for (AuxiliarParameterBindingDescriptor binding : bindings) {
      List<String> missingParameters = findMissingParameters(binding.getValue(), parametersAvailable);
      if (!missingParameters.isEmpty()) {
        results.add(getValidationError(operationAdapterDescriptor, binding.getName(), bindingParameterType, missingParameters));
      }
    }
  }

  private void analyzeBody(List<ValidationResult> results, OperationAdapterDescriptor operationAdapterDescriptor,
                           List<String> parametersAvailable, String bodyExpression) {

    List<String> missingParameters = findMissingParameters(bodyExpression, parametersAvailable);
    if (!missingParameters.isEmpty()) {
      results.add(getValidationError(operationAdapterDescriptor, BODY.getName(), BODY, missingParameters));
    }
  }

  private List<String> findMissingParameters(String expression, List<String> parameters) {
    if (isBlank(expression)) {
      return emptyList();
    }

    List<String> parametersInExpression = new ArrayList<>();
    Matcher matcher = PARAMETER_EXTRACTION_PATTERN.matcher(expression);
    while (matcher.find()) {
      parametersInExpression.add(matcher.group(1));
    }
    return parametersInExpression.stream().filter(p -> !parameters.contains(p)).collect(Collectors.toList());
  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              String binding,
                                              ParameterType parameterType,
                                              List<String> missingParameters) {
    String detail = parameterType.getName()
        + " parameter binding '"
        + binding
        + "' defined in Operation with name: "
        + operationAdapterDescriptor.getOperationId()
        + " has an expression with the following invalid parameters: "
        + join(missingParameters, ",");

    return new ValidationResult(this, detail, operationAdapterDescriptor.getLocation());
  }
}
