/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperation;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.ArrayList;
import java.util.List;

public class TriggerBindingParameterMustExistInApiRule extends DescriptorValidationRule {

  public TriggerBindingParameterMustExistInApiRule() {
    super("Parameter bindings defined in a trigger must exist in the API spec.", EMPTY, WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (!triggerDescriptors.isEmpty()) {
      for (TriggerDescriptor triggerDescriptor : connectorDescriptor.getTriggers()) {

        APIOperationModel operationModel =
            getApiOperation(apiModel, triggerDescriptor.getPath(),
                            triggerDescriptor.getMethod() != null ? triggerDescriptor.getMethod().getName() : null);
        if (operationModel == null) {
          continue;
        }

        if (triggerDescriptor.getParameterBindings() != null) {
          ParameterBindingsDescriptor parameterBindings = triggerDescriptor.getParameterBindings();
          analyzeParameters(results, triggerDescriptor, operationModel, parameterBindings.getHeaders(), HEADER);
          analyzeParameters(results, triggerDescriptor, operationModel, parameterBindings.getQueryParameters(), QUERY);
          analyzeParameters(results, triggerDescriptor, operationModel, parameterBindings.getUriParameters(), URI);
        }
      }
    }

    return results;
  }

  private void analyzeParameters(List<ValidationResult> results, TriggerDescriptor triggerDescriptor,
                                 APIOperationModel operationModel, List<ParameterBindingDescriptor> bindings,
                                 ParameterType bindingParameterType) {
    if (bindings == null) {
      return;
    }

    for (ParameterBindingDescriptor binding : bindings) {
      if (!hasMatchingParameter(operationModel, binding, bindingParameterType)) {
        results.add(getValidationError(triggerDescriptor, binding, bindingParameterType));
      }
    }

  }

  private boolean hasMatchingParameter(APIOperationModel operationModel, ParameterBindingDescriptor binding,
                                       ParameterType parameterType) {
    final List<APIParameterModel> parameters;
    if (parameterType == HEADER) {
      parameters = operationModel.getHeadersModel();
    } else if (parameterType == QUERY) {
      parameters = operationModel.getQueryParamsModel();
    } else {
      parameters = operationModel.getUriParamsModel();
    }

    return parameters.stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(binding.getName()));
  }



  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor,
                                              ParameterBindingDescriptor binding,
                                              ParameterType parameterType) {
    String detail = "'" +
        parameterType.getName()
        + "' parameter binding '"
        + binding.getName()
        + "' defined in Trigger with PATH: "
        + triggerDescriptor.getPath()
        + " and METHOD: "
        + triggerDescriptor.getMethod().getName().toUpperCase()
        + " declares a binding that does not exist in the API spec.";

    return new ValidationResult(this, detail, triggerDescriptor.getLocation());
  }
}
