/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.valueprovider.ValueProviderDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperationModel;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getValueProviderReferenceDescriptors;
import static java.lang.String.format;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

public class ValueProviderReferenceArgumentsOperationParameterExistsRule extends DescriptorValidationRule {

  private static final String ERROR_TEMPLATE = "Argument '%s' uses a parameter that doesn't exist in the API operation: '%s'.";

  public ValueProviderReferenceArgumentsOperationParameterExistsRule() {
    super("All parameter references used as argument for a value provider must exist in the API Operation.",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> validationResults = new ArrayList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        for (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor> reference : getValueProviderReferenceDescriptors(operationDescriptor)) {
          final List<ResolverArgumentDescriptor> arguments = reference.getArguments();

          final List<ValidationResult> errors = arguments.stream()
              .filter(x -> validArgumentValueFormat(x.getValue()))
              .filter(x -> !argumentParameterExistsInOperation(x.getValue(), getApiOperationModel(apiModel, endPointDescriptor,
                                                                                                  operationDescriptor)))
              .map(this::getValidationError)
              .collect(toList());

          validationResults.addAll(errors);
        }
      }
    }

    return validationResults;
  }

  private boolean argumentParameterExistsInOperation(String value, APIOperationModel apiOperationModel) {
    final String parameterType = getParameterTypeReference(value);
    final String parameterName = getParameterReference(value);

    if (parameterType == null || parameterName == null) {
      return false;
    }

    if (parameterType.startsWith(URI.getBinding())) {
      return apiOperationModel.getUriParamsModel().stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName));
    } else if (parameterType.startsWith(QUERY.getBinding())) {
      return apiOperationModel.getQueryParamsModel().stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName));
    } else if (parameterType.startsWith(HEADER.getBinding())) {
      return apiOperationModel.getHeadersModel().stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameterName));
    }

    return false;
  }

  private static ParameterType[] BINDINGS = new ParameterType[] {
      ParameterType.URI,
      ParameterType.QUERY,
      ParameterType.HEADER
  };

  private String getParameterTypeReference(String value) {
    if (isBlank(value)) {
      return null;
    }
    return stream(BINDINGS)
        .map(ParameterType::getBinding)
        .filter(accessorName -> DataWeaveExpressionParser.isBindingUsed(value, accessorName))
        .findFirst()
        .orElse(null);
  }

  private String getParameterReference(String value) {
    if (isBlank(value)) {
      return null;
    }

    if (isBlank(value)) {
      return null;
    }
    return stream(BINDINGS)
        .flatMap((b) -> stream(DataWeaveExpressionParser.selectionsFromBinding(value, b.getBinding())))
        .findFirst()
        .orElse(null);
  }

  private boolean validArgumentValueFormat(String value) {
    if (isBlank(value)) {
      return false;
    }
    return Arrays.stream(BINDINGS).sequential()
        .anyMatch((b) -> DataWeaveExpressionParser.isBindingUsed(value, b.getBinding()));
  }


  private ValidationResult getValidationError(ArgumentDescriptor argumentDescriptor) {
    return new ValidationResult(this,
                                format(ERROR_TEMPLATE, argumentDescriptor.getName(), argumentDescriptor.getValue()),
                                argumentDescriptor.getLocation());
  }
}
