/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.util;

import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElement;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.FieldDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.RequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.valueprovider.ValueProviderDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;

public class ValidationUtils {

  public static EndPointDescriptor getEndpointDescriptor(ConnectorDescriptor connectorDescriptor,
                                                         APIOperationModel apiOperationModel) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .findFirst()
        .orElse(null);
  }

  public static OperationDescriptor getOperationDescriptor(EndPointDescriptor endPointDescriptor,
                                                           APIOperationModel apiOperationModel) {
    return endPointDescriptor.getOperations().stream()
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod()))
        .findFirst().orElse(null);
  }

  public static DescriptorElement getClosestDescriptorElement(ConnectorDescriptor connectorDescriptor,
                                                              APIOperationModel apiOperationModel) {
    final EndPointDescriptor endpointDescriptor = getEndpointDescriptor(connectorDescriptor, apiOperationModel);

    if (endpointDescriptor == null) {
      return connectorDescriptor;
    }

    OperationDescriptor operationDescriptor = getOperationDescriptor(endpointDescriptor, apiOperationModel);
    if (operationDescriptor == null) {
      return endpointDescriptor;
    }

    return operationDescriptor;
  }

  public static EndPointDescriptor getEndpointDescriptor(ConnectorDescriptor connectorDescriptor,
                                                         ConnectorOperation connectorOperation) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(connectorOperation.getPath()))
        .findFirst()
        .orElse(null);
  }

  public static OperationDescriptor getOperationDescriptor(EndPointDescriptor endPointDescriptor,
                                                           ConnectorOperation connectorOperation) {
    return endPointDescriptor.getOperations().stream()
        .filter(x -> x.getMethod().equalsIgnoreCase(connectorOperation.getHttpMethod().name()))
        .findFirst().orElse(null);
  }

  public static DescriptorElement getClosestDescriptorElement(ConnectorDescriptor connectorDescriptor,
                                                              ConnectorOperation connectorOperation) {
    final EndPointDescriptor endpointDescriptor = getEndpointDescriptor(connectorDescriptor, connectorOperation);

    if (endpointDescriptor == null) {
      return connectorDescriptor;
    }

    OperationDescriptor operationDescriptor = getOperationDescriptor(endpointDescriptor, connectorOperation);
    if (operationDescriptor == null) {
      return endpointDescriptor;
    }

    return operationDescriptor;
  }

  public static APIOperationModel getApiOperationModel(APIModel apiModel, ConnectorOperation connectorOperation) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> connectorOperation.getPath().equalsIgnoreCase(x.getPath()))
        .filter(x -> connectorOperation.getHttpMethod().name().equalsIgnoreCase(x.getHttpMethod()))
        .findFirst().orElse(null);
  }

  public static boolean apiOperationIsPresentInConnectorModel(APIOperationModel apiOperationModel,
                                                              ConnectorModel connectorModel) {
    return connectorModel.getOperations().stream()
        .anyMatch(op -> op.getPath().equalsIgnoreCase(apiOperationModel.getPath())
            && op.getHttpMethod().name().equalsIgnoreCase(apiOperationModel.getHttpMethod()));
  }

  public static APIOperationModel getApiOperation(APIModel apiModel, String path, String method) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path))
        .filter(x -> x.getHttpMethod().equalsIgnoreCase(method))
        .findFirst().orElse(null);
  }

  public static OperationDescriptor getOperationDescriptor(ConnectorDescriptor connectorDescriptor,
                                                           APIOperationModel apiOperation) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperation.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperation.getHttpMethod()))
        .findFirst().orElse(null);
  }

  public static List<ResolverExpressionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderDescriptors(ConnectorDescriptor connectorDescriptor) {

    List<ResolverExpressionDescriptor<ValueProviderDefinitionDescriptor>> valueProviderDescriptors =
        connectorDescriptor.getEndpoints().stream()
            .flatMap(x -> x.getOperations().stream())
            .flatMap(x -> getValueProviderDescriptors(x).stream())
            .filter(Objects::nonNull)
            .collect(toList());

    connectorDescriptor.getEndpoints().stream()
        .flatMap(x -> x.getOperations().stream())
        .flatMap(x -> getValueProviderBodyLevelDescriptors(x).stream())
        .filter(Objects::nonNull)
        .forEach(valueProviderDescriptors::add);

    connectorDescriptor.getOperationAdapterDescriptors().stream()
        .flatMap(x -> getValueProviderDescriptors(x).stream())
        .filter(Objects::nonNull)
        .forEach(valueProviderDescriptors::add);

    return valueProviderDescriptors;
  }

  public static List<ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderReferenceDescriptors(
                                                                                                                          ConnectorDescriptor connectorDescriptor) {

    List<ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>> valueProviderReferenceDescriptors = new ArrayList<>();

    // Endpoint references
    valueProviderReferenceDescriptors.addAll(
                                             connectorDescriptor.getEndpoints().stream()
                                                 .flatMap(x -> x.getOperations().stream())
                                                 .flatMap(x -> getValueProviderReferenceDescriptors(x).stream())
                                                 .collect(toList()));
    // Custom operation references
    valueProviderReferenceDescriptors.addAll(
                                             connectorDescriptor.getOperationAdapterDescriptors().stream()
                                                 .flatMap(x -> getValueProviderReferenceDescriptors(x).stream())
                                                 .collect(toList()));

    return valueProviderReferenceDescriptors;
  }

  public static List<ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderInlineDescriptors(
                                                                                                                        ConnectorDescriptor connectorDescriptor) {

    List<ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>> valueProviderReferenceDescriptors = new ArrayList<>();

    // Endpoint references
    valueProviderReferenceDescriptors.addAll(
                                             connectorDescriptor.getEndpoints().stream()
                                                 .flatMap(x -> x.getOperations().stream())
                                                 .flatMap(x -> getValueProviderInlineDescriptors(x).stream())
                                                 .collect(toList()));
    // Custom operation references
    valueProviderReferenceDescriptors.addAll(
                                             connectorDescriptor.getOperationAdapterDescriptors().stream()
                                                 .flatMap(x -> getValueProviderInlineDescriptors(x).stream())
                                                 .collect(toList()));

    return valueProviderReferenceDescriptors;
  }

  public static List<ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderBodyLevelReferenceDescriptors(
                                                                                                                                   ConnectorDescriptor connectorDescriptor) {

    return connectorDescriptor.getEndpoints().stream()
        .flatMap(x -> x.getOperations().stream())
        .flatMap(x -> getValueProviderBodyLevelReferenceDescriptors(x).stream())
        .collect(toList());
  }

  public static List<ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderBodyLevelInlineDescriptors(
                                                                                                                                 ConnectorDescriptor connectorDescriptor) {

    return connectorDescriptor.getEndpoints().stream()
        .flatMap(x -> x.getOperations().stream())
        .flatMap(x -> getValueProviderBodyLevelInlineDescriptors(x).stream())
        .collect(toList());
  }

  public static List<ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderReferenceDescriptors(
                                                                                                                          OperationDescriptor operationDescriptor) {

    return getAllParameters(operationDescriptor.getExpects()).stream()
        .map(ParameterDescriptor::getValueProvider)
        .filter(x -> x instanceof ResolverReferenceDescriptor<?>)
        .map(x -> (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>) x)
        .collect(toList());
  }

  public static List<ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderInlineDescriptors(
                                                                                                                        OperationDescriptor operationDescriptor) {

    return getAllParameters(operationDescriptor.getExpects()).stream()
        .map(ParameterDescriptor::getValueProvider)
        .filter(x -> x instanceof ResolverDefinitionDescriptor<?>)
        .map(x -> (ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>) x)
        .collect(toList());
  }

  public static List<ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderReferenceDescriptors(
                                                                                                                          OperationAdapterDescriptor operationAdapterDescriptor) {

    return operationAdapterDescriptor.getParameters().stream()
        .map(AuxiliarParameterDescriptor::getValueProvider)
        .filter(x -> x instanceof ResolverReferenceDescriptor<?>)
        .map(x -> (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>) x)
        .collect(toList());
  }

  public static List<ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderInlineDescriptors(
                                                                                                                        OperationAdapterDescriptor operationAdapterDescriptor) {

    return operationAdapterDescriptor.getParameters().stream()
        .map(AuxiliarParameterDescriptor::getValueProvider)
        .filter(x -> x instanceof ResolverDefinitionDescriptor<?>)
        .map(x -> (ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>) x)
        .collect(toList());
  }

  public static List<ResolverExpressionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderDescriptors(OperationDescriptor operationDescriptor) {

    return getAllParameters(operationDescriptor.getExpects()).stream()
        .map(ParameterDescriptor::getValueProvider)
        .collect(toList());
  }

  public static List<ResolverExpressionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderDescriptors(OperationAdapterDescriptor operationDescriptor) {

    return operationDescriptor.getParameters().stream()
        .map(AuxiliarParameterDescriptor::getValueProvider)
        .collect(toList());
  }

  public static List<ResolverReferenceDescriptor<SampleDataDefinitionDescriptor>> getSampleDataReferenceDescriptors(
                                                                                                                    ConnectorDescriptor connectorDescriptor) {

    List<ResolverReferenceDescriptor<SampleDataDefinitionDescriptor>> sampleDataReferenceDescriptors = new ArrayList<>();

    // Endpoint references
    sampleDataReferenceDescriptors.addAll(
                                          connectorDescriptor.getEndpoints().stream()
                                              .flatMap(x -> x.getOperations().stream())
                                              .map(x -> getSampleDataReferenceDescriptor(x.getSampleDataExpressionDescriptor()))
                                              .filter(x -> x != null)
                                              .collect(toList()));
    // Custom operation references
    sampleDataReferenceDescriptors.addAll(
                                          connectorDescriptor.getOperationAdapterDescriptors().stream()
                                              .map(x -> getSampleDataReferenceDescriptor(x.getSampleDataExpressionDescriptor()))
                                              .filter(x -> x != null)
                                              .collect(toList()));

    return sampleDataReferenceDescriptors;
  }

  public static List<ResolverDefinitionDescriptor<SampleDataDefinitionDescriptor>> getSampleDataInlineDescriptors(
                                                                                                                  ConnectorDescriptor connectorDescriptor) {

    List<ResolverDefinitionDescriptor<SampleDataDefinitionDescriptor>> sampleDataReferenceDescriptors = new ArrayList<>();

    // Endpoint references
    sampleDataReferenceDescriptors.addAll(
                                          connectorDescriptor.getEndpoints().stream()
                                              .flatMap(x -> x.getOperations().stream())
                                              .map(x -> getSampleDataInlineDescriptor(x.getSampleDataExpressionDescriptor()))
                                              .filter(x -> x != null)
                                              .collect(toList()));
    // Custom operation references
    sampleDataReferenceDescriptors.addAll(
                                          connectorDescriptor.getOperationAdapterDescriptors().stream()
                                              .map(x -> getSampleDataInlineDescriptor(x.getSampleDataExpressionDescriptor()))
                                              .filter(x -> x != null)
                                              .collect(toList()));

    return sampleDataReferenceDescriptors;
  }

  public static ResolverReferenceDescriptor<SampleDataDefinitionDescriptor> getSampleDataReferenceDescriptor(
                                                                                                             ResolverExpressionDescriptor<SampleDataDefinitionDescriptor> sampleDataExpressionDescriptor) {

    if (sampleDataExpressionDescriptor instanceof ResolverReferenceDescriptor<?>) {
      return (ResolverReferenceDescriptor<SampleDataDefinitionDescriptor>) sampleDataExpressionDescriptor;
    }
    return null;
  }

  public static ResolverDefinitionDescriptor<SampleDataDefinitionDescriptor> getSampleDataInlineDescriptor(
                                                                                                           ResolverExpressionDescriptor<SampleDataDefinitionDescriptor> sampleDataExpressionDescriptor) {

    if (sampleDataExpressionDescriptor instanceof ResolverDefinitionDescriptor<?>) {
      return (ResolverDefinitionDescriptor<SampleDataDefinitionDescriptor>) sampleDataExpressionDescriptor;
    }
    return null;
  }

  public static List<ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderBodyLevelReferenceDescriptors(
                                                                                                                                   OperationDescriptor operationDescriptor) {

    return getAllFields(operationDescriptor.getExpects()).stream()
        .map(FieldDescriptor::getValueProviders)
        .filter(x -> x instanceof ResolverReferenceDescriptor<?>)
        .map(x -> (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor>) x)
        .collect(toList());
  }

  public static List<ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderBodyLevelInlineDescriptors(
                                                                                                                                 OperationDescriptor operationDescriptor) {

    return getAllFields(operationDescriptor.getExpects()).stream()
        .map(FieldDescriptor::getValueProviders)
        .filter(x -> x instanceof ResolverDefinitionDescriptor<?>)
        .map(x -> (ResolverDefinitionDescriptor<ValueProviderDefinitionDescriptor>) x)
        .collect(toList());
  }

  public static List<ResolverExpressionDescriptor<ValueProviderDefinitionDescriptor>> getValueProviderBodyLevelDescriptors(OperationDescriptor operationDescriptor) {

    return getAllFields(operationDescriptor.getExpects()).stream()
        .map(FieldDescriptor::getValueProviders)
        .collect(toList());
  }

  public static APIOperationModel getApiOperationModel(APIModel apiModel, EndPointDescriptor endpointDescriptor,
                                                       OperationDescriptor operationDescriptor) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(endpointDescriptor.getPath()))
        .filter(x -> x.getHttpMethod().equalsIgnoreCase(operationDescriptor.getMethod()))
        .findFirst().orElse(null);
  }

  public static <T> Predicate<T> distinctByKey(Function<? super T, Object> keyExtractor) {
    Map<Object, Boolean> map = new ConcurrentHashMap<>();
    return t -> map.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
  }

  private static List<ParameterDescriptor> getAllParameters(RequestDescriptor requestDescriptor) {
    List<ParameterDescriptor> parameters = new ArrayList<>();

    if (requestDescriptor != null) {
      parameters.addAll(requestDescriptor.getUriParameter());
      parameters.addAll(requestDescriptor.getQueryParameter());
      parameters.addAll(requestDescriptor.getHeader());
    }

    return parameters;
  }

  private static List<FieldDescriptor> getAllFields(RequestDescriptor requestDescriptor) {
    List<FieldDescriptor> fields = new ArrayList<>();

    if (requestDescriptor != null && requestDescriptor.getBody() != null) {
      fields.addAll(requestDescriptor.getBody().getFieldDescriptors());
    }

    return fields;
  }


}
