/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.api;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.ApiParser.parseModel;
import static com.mulesoft.connectivity.rest.sdk.templating.TemplateEntity.getTemplateEngine;
import static java.lang.String.format;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.api.ApiConnectorModelLoader;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorConnectorModelLoader;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.exception.DescriptorParsingException;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParser;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationEngine;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;


public class RestSdk {

  private final File apiSpec;
  private final String rootDir;
  private final RestSdkRunConfiguration runConfiguration;
  private final List<File> descriptors = new ArrayList<>();

  private ConnectorModel connectorModel;

  public RestSdk(File apiSpec, String rootDir, RestSdkRunConfiguration runConfiguration) {
    this.apiSpec = apiSpec;
    this.rootDir = rootDir;
    this.runConfiguration = runConfiguration;
  }

  public RestSdk descriptor(File descriptor) {
    this.descriptors.add(descriptor);
    return this;
  }

  public ConnectorModel getConnectorModel(ValidationEngine validationEngine)
      throws DescriptorParsingException, ModelGenerationException {

    List<ConnectorDescriptor> descriptorModels = getDescriptorModels();

    APIModel apiModel = parseModel(apiSpec, rootDir, runConfiguration.skipValidations());

    validationEngine.validateApi(apiModel);

    ConnectorModelBuilder connectorModelBuilder = ApiConnectorModelLoader.load(apiModel, new ConnectorModelBuilder());

    for (ConnectorDescriptor descriptorModel : descriptorModels) {
      validationEngine.validateDescriptor(apiModel, descriptorModel, connectorModelBuilder);

      checkResults(validationEngine);

      DescriptorConnectorModelLoader.load(descriptorModel, connectorModelBuilder);

      validationEngine.validateConnectorModelBuilderStep(connectorModelBuilder);
    }

    validationEngine.validateConnectorModelBuilder(connectorModelBuilder);
    checkResults(validationEngine);

    postProcess(connectorModelBuilder, apiModel);
    checkResults(validationEngine);

    ConnectorModel connectorModel = connectorModelBuilder.build();
    validationEngine.validateConnectorModel(apiModel, connectorModel);

    if (!runConfiguration.regenerateMode()) {
      validationEngine.validateConnectorModelCreateConnectorMojo(connectorModel);
    }

    checkResults(validationEngine);

    this.connectorModel = connectorModel;
    return connectorModel;
  }

  private void checkResults(ValidationEngine validationEngine) throws ModelGenerationException {
    if (!validationEngine.isSuccess()) {
      validationEngine.logResults();
      throw new ModelGenerationException(format("There are %s validation errors and %s warnings.",
                                                validationEngine.countResults(ERROR),
                                                validationEngine.countResults(WARN)));
    }
  }

  private void postProcess(ConnectorModelBuilder connectorModelBuilder, APIModel apiModel) {
    // do nothing if it was processed ok or overriden by the user in the descriptor
    if (!connectorModelBuilder.getBaseUriBuilder().hasUri()) {
      Set<String> uriSet = connectorModelBuilder.getOperationBuilders().stream()
          .filter(ob -> ob.isIgnored() == null || !ob.isIgnored())
          .map(OperationBuilder::getBaseUris)
          .flatMap(List::stream)
          .collect(Collectors.toSet());
      // Only try to fix Null URI if all operations have the same URI
      if (uriSet.size() == 1) {
        connectorModelBuilder.getBaseUriBuilder().setUri(uriSet.iterator().next());
      }
    }
  }

  public void generateConnectorSources(Path outputDir, RestSdkRunConfiguration runConfiguration)
      throws Exception {

    ValidationEngine validationEngine = new ValidationEngine();

    if (this.connectorModel == null) {
      getConnectorModel(validationEngine);
    }

    getTemplateEngine(connectorModel, descriptors, runConfiguration, outputDir)
        .applyTemplates();

    validationEngine.validateFiles(outputDir, runConfiguration);

    checkResults(validationEngine);
    validationEngine.logResults();
  }

  private List<ConnectorDescriptor> getDescriptorModels() throws DescriptorParsingException {

    List<ConnectorDescriptor> descriptorModels = new ArrayList<>();
    DescriptorParser descriptorParser = new DescriptorParser();

    for (File descriptorFile : descriptors) {
      descriptorModels.add(descriptorParser.parseConnectorDescriptor(descriptorFile));
    }

    return descriptorModels;
  }
}
