/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.NOT_SUPPORTED;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;

public class AtLeastOneSupportedSecuritySchemeRule extends ConnectorModelValidationRule {

  // R004
  public AtLeastOneSupportedSecuritySchemeRule() {
    super("At least one of the declared security schemes must be supported",
          "When the API spec declares security, at least one of the declared security schemes must be supported.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    return apiModel.getOperationsModel().stream()
        .filter(opm -> ValidationUtils.apiOperationIsPresentInConnectorModel(opm, connectorModel))
        .filter(this::allSecuritySchemesUnsupported)
        .map(this::getValidationError)
        .collect(toList());
  }

  private boolean allSecuritySchemesUnsupported(APIOperationModel apiOperationModel) {
    return !apiOperationModel.getSecuritySchemesModel().isEmpty() &&
        apiOperationModel.getSecuritySchemesModel().stream().allMatch(x -> x.getSecuritySchemeType().equals(NOT_SUPPORTED));
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String detail =
        "API Operation with PATH: "
            + apiOperationModel.getPath()
            + " and METHOD: "
            + apiOperationModel.getHttpMethod()
            + " declares unsupported security schemes";

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
