/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class OperationIdentifierScriptMustNotRepeatRule extends ConnectorModelValidationRule {

  public OperationIdentifierScriptMustNotRepeatRule() {
    super("Operation identifiers must not be repeated",
          "There are operations with the same 'operationIdentifier'",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    Map<String, List<ConnectorOperation>> operationIdentifiers = new HashMap<>();

    for (ConnectorOperation op : connectorModel.getOperations()) {
      if (!operationIdentifiers.containsKey(op.getOperationIdentifier())) {
        operationIdentifiers.put(op.getOperationIdentifier(), new ArrayList<>());
      }
      operationIdentifiers.get(op.getOperationIdentifier()).add(op);
    }

    List<ValidationResult> errors = operationIdentifiers.entrySet().stream()
        .filter(entry -> entry.getValue().size() > 1)
        .map(entry -> getValidationErrors(entry.getKey(), entry.getValue()))
        .flatMap(Collection::stream)
        .collect(Collectors.toList());

    return errors;
  }

  private List<ValidationResult> getValidationErrors(String collideOperationIdentifier, List<ConnectorOperation> ops) {
    List<ValidationResult> partialErrors = ops.stream().map(op -> {
      String detail;
      if (op.isAdapter()) {
        detail = format("The operation [%s] identifier is already defined.", op.getOperationIdentifier());
      } else {
        String operationName = op.getHttpMethod() + ":" + op.getPath();
        detail = format("The operation [%s] has an 'operationIdentifier: %s' value that's already defined.", operationName,
                        collideOperationIdentifier);
      }

      DescriptorElementLocation loc = DescriptorElementLocation.builder().empty();
      return new ValidationResult(this, detail, loc);
    })
        .collect(Collectors.toList());
    return partialErrors;
  }
}
