/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

public abstract class AbstractResolverReferenceParametersMustReferencedRule<T extends ResolverDefinitionDescriptor<T>>
    extends DescriptorValidationRule {

  private final String ERROR_TEMPLATE;

  public AbstractResolverReferenceParametersMustReferencedRule(String resolverType) {
    super(format("When using a %s it must reference all the parameters in the resolver declaration", resolverType), EMPTY,
          ERROR);
    ERROR_TEMPLATE = new StringBuilder()
        .append("Parameter '%s' is not referenced in the '%s' ")
        .append(resolverType)
        .append(" reference.")
        .toString();
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> validationResults = new ArrayList<>();

    for (ResolverReferenceDescriptor<T> reference : getResolverReferenceDescriptors(connectorDescriptor)) {
      final List<ResolverArgumentDescriptor> arguments = reference.getArguments();
      final List<ResolverParameterDescriptor> parameters = reference.getDeclaration().getParameters();

      // Each parameter has a defined argument validation
      final List<ValidationResult> errors = parameters.stream()
          .filter(x -> arguments.stream().noneMatch(y -> y.getName().equalsIgnoreCase(x.getName())))
          .map(x -> getValidationError(x, reference.getDeclaration()))
          .collect(toList());

      validationResults.addAll(errors);
    }

    return validationResults;
  }

  protected abstract List<ResolverReferenceDescriptor<T>> getResolverReferenceDescriptors(ConnectorDescriptor connectorDescriptor);

  private ValidationResult getValidationError(ResolverParameterDescriptor parameterDescriptor,
                                              ResolverDeclarationDescriptor<?> declaration) {
    return new ValidationResult(this,
                                format(ERROR_TEMPLATE, parameterDescriptor.getName(), declaration.getName()),
                                parameterDescriptor.getLocation());
  }
}
