/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterRequestBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class OperationAdapterBindingParameterWithValueMustNotBeIgnoredRule extends DescriptorValidationRule {

  public OperationAdapterBindingParameterWithValueMustNotBeIgnoredRule() {
    super("Parameter bindings with value cannot be ignored.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    List<OperationAdapterDescriptor> operationAdapterDescriptors = connectorDescriptor.getOperationAdapterDescriptors();
    if (!operationAdapterDescriptors.isEmpty()) {
      for (OperationAdapterDescriptor operationAdapterDescriptor : operationAdapterDescriptors) {
        AuxiliarParameterRequestBindingsDescriptor requestBindings = operationAdapterDescriptor.getRequestBindings();
        if (requestBindings != null) {
          analyzeParameters(results, operationAdapterDescriptor, requestBindings.getHeaders(), HEADER);
          analyzeParameters(results, operationAdapterDescriptor, requestBindings.getQueryParameters(), QUERY);
          analyzeParameters(results, operationAdapterDescriptor, requestBindings.getUriParameters(), URI);
        }
      }
    }

    return results;
  }

  private void analyzeParameters(List<ValidationResult> results, OperationAdapterDescriptor operationAdapterDescriptor,
                                 List<AuxiliarParameterBindingDescriptor> bindings, ParameterType bindingParameterType) {

    for (AuxiliarParameterBindingDescriptor binding : bindings) {
      if (binding.getIgnored() != null && binding.getIgnored() && isNotEmpty(binding.getValue())) {
        results.add(getValidationError(operationAdapterDescriptor, binding, bindingParameterType));
      }
    }

  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              AuxiliarParameterBindingDescriptor binding,
                                              ParameterType parameterType) {
    String detail = parameterType.getName()
        + " parameter binding '"
        + binding.getName()
        + "' defined in Operation with name: "
        + operationAdapterDescriptor.getOperationId()
        + " is ignored and has a value defined.";

    return new ValidationResult(this, detail, operationAdapterDescriptor.getLocation());
  }
}
