/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class SecuritySchemeMustBeSupportedByTheApiRule extends DescriptorValidationRule {

  public SecuritySchemeMustBeSupportedByTheApiRule() {
    super("All security schemes used in the descriptor must be supported by the Api",
          EMPTY,
          ERROR);
  }


  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> validationsResults = new ArrayList<>();

    DescriptorElementLocation descriptorElementLocation = connectorDescriptor.getLocation();

    Set<String> descriptorSecuritySchemas = connectorDescriptor.getSecurity().stream()
        .map(SecuritySchemeBaseDescriptor::getName)
        .collect(Collectors.toSet());

    Set<String> securitySchemasSupportedByApi = apiModel.getOperationsModel().stream()
        .flatMap(x -> x.getSecuritySchemesModel().stream())
        .map(x -> x.getName())
        .collect(Collectors.toSet());

    descriptorSecuritySchemas.removeAll(securitySchemasSupportedByApi);


    if (descriptorSecuritySchemas.size() > 0) {
      descriptorSecuritySchemas
          .forEach(x -> validationsResults.add(getValidationError(x, securitySchemasSupportedByApi, descriptorElementLocation)));
    }

    return validationsResults;
  }

  private ValidationResult getValidationError(String unsupportedScheme,
                                              Set<String> securitySchemasSupportedByApi,
                                              DescriptorElementLocation descriptorElementLocation) {
    String detail =
        "The Security Scheme with name: '"
            + unsupportedScheme +
            "' is not declared in the apiSpec. The Security Schemes supported by the API are: '"
            + String.join("', '", securitySchemasSupportedByApi)
            + "'.";

    return new ValidationResult(this, detail, descriptorElementLocation);
  }


}
