/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorSecurityType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.OAUTH2_AUTHORIZATION_CODE;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.OAUTH2_CLIENT_CREDENTIALS;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;

public class SecurityTypeMustBeTheSameRule extends DescriptorValidationRule {

  public SecurityTypeMustBeTheSameRule() {
    super("The security schemes must match the security type in the descriptor and in the apiSpec.",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationsResults = new ArrayList<>();
    for (APIOperationModel apiOperation : apiModel.getOperationsModel()) {
      for (APISecuritySchemeModel apiSecuritySchemeModel : apiOperation.getSecuritySchemesModel())
        for (SecuritySchemeBaseDescriptor descriptorSecurityScheme : connectorDescriptor.getSecurity()) {
          if (apiSecuritySchemeModel.getName().equals(descriptorSecurityScheme.getName())) {
            if (!haveSameType(apiSecuritySchemeModel.getSecuritySchemeType(), descriptorSecurityScheme.getType())) {
              ValidationResult validationResult = getValidationError(apiSecuritySchemeModel, descriptorSecurityScheme);
              if (validationsResults.stream().noneMatch(x -> x.getDetail().equals(validationResult.getDetail()))) {
                validationsResults.add(getValidationError(apiSecuritySchemeModel, descriptorSecurityScheme));
              }
            }
          }
        }
    }
    return validationsResults;
  }

  public boolean haveSameType(APISecuritySchemeType apiType, DescriptorSecurityType typeDescriptor) {
    if (apiType.equals(OAUTH2_AUTHORIZATION_CODE) && typeDescriptor.equals(DescriptorSecurityType.OAUTH2)
        || (apiType.equals(OAUTH2_CLIENT_CREDENTIALS) && typeDescriptor.equals(DescriptorSecurityType.OAUTH2))) {
      return true;
    }
    return apiType.name().equals(typeDescriptor.name());
  }

  private ValidationResult getValidationError(APISecuritySchemeModel apiSecuritySchemeModel,
                                              SecuritySchemeBaseDescriptor securitySchemeDescriptor) {
    String detail =
        "API Security scheme with name: "
            + apiSecuritySchemeModel.getName() +
            " and type: "
            + apiSecuritySchemeModel.getSecuritySchemeType().name() +
            " was declared in the descriptor with this type: "
            + securitySchemeDescriptor.getType()
            + ". The type must be the same.";

    return new ValidationResult(this, detail, securitySchemeDescriptor.getLocation());
  }

}
