/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.TriggerNode;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import static java.util.stream.Collectors.joining;

import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.apache.commons.lang3.StringUtils.EMPTY;

public class TriggerCircularDependenciesRule extends DescriptorValidationRule {

  public TriggerCircularDependenciesRule() {
    super("A circular dependency has been found in base triggers dependencies.", EMPTY, ERROR);
  }

  private TriggerNode getOrCreate(TriggerDescriptor triggerDescriptor, List<TriggerNode> createdTriggersNodesList) {
    if (triggerDescriptor != null) {
      TriggerNode node =
          createdTriggersNodesList.stream().filter(x -> x.getTrigger().getName().equals(triggerDescriptor.getName()))
              .findFirst().orElse(null);
      if (node == null) {
        node = new TriggerNode(triggerDescriptor);
        createdTriggersNodesList.add(node);
      }
      return node;
    }
    // If return null, build will fail for previous rule BaseTriggerMustExistRule
    return null;
  }

  private void addParent(TriggerNode triggerNode, TriggerDescriptor triggerDescriptor, ConnectorDescriptor connectorDescriptor,
                         List<TriggerNode> createdTriggersNodesList) {
    if (triggerDescriptor.getBase() != null) {
      TriggerNode parent = getOrCreate(
                                       connectorDescriptor.getTriggers().stream()
                                           .filter(x -> x.getName().equals(triggerDescriptor.getBase())).findFirst().orElse(null),
                                       createdTriggersNodesList);
      triggerNode.setParent(parent);
    }
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<TriggerNode> createdTriggersNodesList = new ArrayList<>();
    final List<ValidationResult> results = new ArrayList<>();

    List<TriggerDescriptor> triggerDescriptors = connectorDescriptor.getTriggers();
    if (!triggerDescriptors.isEmpty()) {
      // Load node list
      for (TriggerDescriptor triggerDescriptor : connectorDescriptor.getTriggers()) {
        TriggerNode node = getOrCreate(triggerDescriptor, createdTriggersNodesList);
        addParent(node, triggerDescriptor, connectorDescriptor, createdTriggersNodesList);
      }

      // Check circular dependency
      for (TriggerNode triggerNode : createdTriggersNodesList) {
        if (!triggerNode.isVisited()) {
          Map<String, Boolean> usedTriggers = new HashMap<>();
          usedTriggers.put(triggerNode.getTrigger().getName(), true);
          TriggerNode pointer = triggerNode.getParent();
          triggerNode.setVisited(true);
          while (pointer != null) {
            pointer.setVisited(true);
            if (usedTriggers.get(pointer.getTrigger().getName()) != null) {
              // Circular dependency
              results.add(getValidationError(usedTriggers.keySet().stream().collect(joining(", ")), pointer.getTrigger()));
              break;
            } else {
              usedTriggers.put(pointer.getTrigger().getName(), true);
              pointer = pointer.getParent();
            }
          }
        }
      }
    }
    return results;
  }

  private ValidationResult getValidationError(String triggers, TriggerDescriptor triggerDescriptor) {
    String detail = "Triggers involved: " + triggers;
    return new ValidationResult(this, detail, triggerDescriptor.getLocation());
  }
}
