/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterRequestBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import org.apache.commons.lang3.StringUtils;

public class OperationAdapterRequestBindingParametersMustExistsRule extends DescriptorConnectorModelBuilderValidationRule {

  public OperationAdapterRequestBindingParametersMustExistsRule() {
    super("Request parameter bindings defined in operation adapter must exist in the API spec.",
          EMPTY,
          WARN);
  }

  private OperationBuilder getBaseEndpoint(OperationAdapterDescriptor operationAdapterDescriptor,
                                           ConnectorModelBuilder connectorModelBuilder) {
    return connectorModelBuilder.getOperationBuildersByOperationId(operationAdapterDescriptor.getBaseEndpoint()).get();
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor,
                                         ConnectorModelBuilder connectorModelBuilder) {
    final List<ValidationResult> results = new ArrayList<>();

    List<OperationAdapterDescriptor> operationAdapterDescriptors = connectorDescriptor.getOperationAdapterDescriptors()
        .stream().filter(x -> StringUtils.isBlank(x.getFqn())).collect(Collectors.toList());
    for (OperationAdapterDescriptor operationAdapterDescriptor : operationAdapterDescriptors) {
      AuxiliarParameterRequestBindingsDescriptor requestBindings = operationAdapterDescriptor.getRequestBindings();
      OperationBuilder operationBuilder = getBaseEndpoint(operationAdapterDescriptor, connectorModelBuilder);
      if (requestBindings != null) {
        analyzeParameters(results, operationBuilder.getHeaders(), requestBindings.getHeaders(), HEADER,
                          operationAdapterDescriptor);
        analyzeParameters(results, operationBuilder.getQueryParameters(), requestBindings.getQueryParameters(), QUERY,
                          operationAdapterDescriptor);
        analyzeParameters(results, operationBuilder.getUriParameters(), requestBindings.getUriParameters(), URI,
                          operationAdapterDescriptor);
      }
    }

    return results;
  }

  private void analyzeParameters(List<ValidationResult> results, List<ParameterBuilder> baseParameters,
                                 List<AuxiliarParameterBindingDescriptor> bindings, ParameterType bindingParameterType,
                                 OperationAdapterDescriptor operationAdapterDescriptor) {

    if (bindings.isEmpty()) {
      return;
    }

    for (AuxiliarParameterBindingDescriptor binding : bindings) {
      if (baseParameters.isEmpty()
          || baseParameters.stream().noneMatch(x -> x.getExternalName().equals(binding.getName()))) {
        results.add(getValidationError(operationAdapterDescriptor, binding, bindingParameterType));
      }
    }

  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              AuxiliarParameterBindingDescriptor binding,
                                              ParameterType parameterType) {
    String detail = parameterType.getName()
        + " parameter binding '"
        + binding.getName()
        + "' defined in Operation with name: "
        + operationAdapterDescriptor.getOperationId()
        + " declares a request binding that does not exist in the API spec.";

    return new ValidationResult(this, detail, operationAdapterDescriptor.getLocation());
  }
}
