/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.files;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.VALIDATION_ERROR_MESSAGE_TEMPLATE;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.isRefinementFile;
import static java.lang.String.format;
import static java.nio.file.Files.exists;
import static java.nio.file.Files.walk;
import static java.nio.file.Paths.get;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.FilesValidationRule;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import org.slf4j.Logger;

public class FileInOverrideMustExistInGeneratedCodeRule extends FilesValidationRule {

  private static final Logger LOGGER = getLogger(FileInOverrideMustExistInGeneratedCodeRule.class);

  public FileInOverrideMustExistInGeneratedCodeRule() {
    super("File In Override folder must exist in generated code", EMPTY, WARN);
  }

  @Override
  public List<ValidationResult> validate(Path outputDir, RestSdkRunConfiguration runConfiguration) {
    if (!runConfiguration.regenerateMode()) {
      return emptyList();
    }

    try {
      Path overrideDir = outputDir.resolve(runConfiguration.getOverrideDir());

      if (overrideDir.toFile().exists()) {
        return walk(overrideDir)
            .filter(Files::isRegularFile)
            .filter(x -> !isRefinementFile(x)) // Avoid double WARNs, as RefinementsGoInRefinementFolderRule covers this.
            .filter(file -> !isValidOverrideFile(file, outputDir, runConfiguration))
            .map(this::getValidationResult)
            .collect(toList());
      }

    } catch (IOException e) {
      String message = format(VALIDATION_ERROR_MESSAGE_TEMPLATE, e.getMessage());
      LOGGER.error(message, e);
      return emptyList();
    }

    return emptyList();
  }

  private ValidationResult getValidationResult(Path file) {
    String detail = format("File '%s' is not overriding a generated file.", file.toString());

    return new ValidationResult(this, detail, null);
  }

  private boolean isValidOverrideFile(Path file, Path outputDir, RestSdkRunConfiguration runConfiguration) {
    String fileRelativePath = outputDir.relativize(file).toString().replace(runConfiguration.getOverrideDir(), EMPTY);
    String generatedFile = outputDir.resolve(runConfiguration.getIgnoredDir()).toString().concat(fileRelativePath);
    return exists(get(generatedFile));
  }
}
