/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.api;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.distinctByKey;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.OAUTH2_AUTHORIZATION_CODE;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ApiValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;

import java.util.ArrayList;
import java.util.List;

public class OauthAuthorizationUrlMustExistRule extends ApiValidationRule {

  public OauthAuthorizationUrlMustExistRule() {
    super("'AuthorizationUrl' should have a value in Oauth 2.0 Authorization Code authentication",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel) {
    List<ValidationResult> validationResults = new ArrayList<>();
    List<APISecuritySchemeModel> listSecuritySchemeModels = apiModel.getOperationsModel().stream()
        .map(APIOperationModel::getSecuritySchemesModel)
        .flatMap(List::stream)
        .filter(distinctByKey(APISecuritySchemeModel::getName))
        .collect(toList());

    for (APISecuritySchemeModel apiSecuritySchemeModel : listSecuritySchemeModels) {
      if (OAUTH2_AUTHORIZATION_CODE.equals(apiSecuritySchemeModel.getSecuritySchemeType()))
        if (apiSecuritySchemeModel.getAuthorizationUri() == null) {
          validationResults.add(getValidationError(apiSecuritySchemeModel.getName()));
        }
    }

    return validationResults;
  }

  private ValidationResult getValidationError(String securityScheme) {
    return new ValidationResult(this, "In the APISpec, security scheme with name " + securityScheme
        + ", 'AuthorizationUrl' was not declared.", DescriptorElementLocation.builder().empty());
  }
}
