/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class ParametersIdentifiersMustNotRepeatRule extends ConnectorModelValidationRule {

  public ParametersIdentifiersMustNotRepeatRule() {
    super("Parameter's identifier must not be repeated within the same operation.",
          "The identifier of some parameters, are colliding.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    List<ValidationResult> validationResults = connectorModel.getOperations()
        .stream()
        .map(this::validateParametersIdentifiers)
        .flatMap(List::stream)
        .collect(Collectors.toList());
    return validationResults;
  }

  private List<ValidationResult> validateParametersIdentifiers(ConnectorOperation operation) {
    Map<String, List<Parameter>> parametersIdentifiers = new HashMap<>();

    Consumer<Parameter> collisionSearcher = parameter -> {
      if (!parametersIdentifiers.containsKey(parameter.getParameterIdentifier())) {
        parametersIdentifiers.put(parameter.getParameterIdentifier(), new ArrayList<>());
      }
      parametersIdentifiers.get(parameter.getParameterIdentifier()).add(parameter);
    };
    operation.getUriParameters().forEach(collisionSearcher);
    operation.getHeaders().forEach(collisionSearcher);
    operation.getQueryParameters().forEach(collisionSearcher);

    List<ValidationResult> validationResults = parametersIdentifiers.entrySet().stream()
        .filter(entry -> entry.getValue().size() > 1)
        .map(entry -> getValidationError(entry.getKey(), operation, entry.getValue()))
        .collect(Collectors.toList());
    return validationResults;
  }

  private ValidationResult getValidationError(String parameterIdentifier, ConnectorOperation op, List<Parameter> parameters) {
    List<String> uriParams = new ArrayList<>();
    List<String> headerParams = new ArrayList<>();
    List<String> queryParams = new ArrayList<>();

    parameters.forEach(p -> {
      switch (p.getParameterType()) {
        case URI:
          uriParams.add(p.getExternalName());
          break;
        case QUERY:
          queryParams.add(p.getExternalName());
          break;
        case HEADER:
          headerParams.add(p.getExternalName());
          break;
        default:
          break;
      }
    });

    String uriParamsBlock = getListWithNamespaceOrEmpty(uriParams, "URI params");
    String headersParamsBlock = getListWithNamespaceOrEmpty(headerParams, "headers");
    String queryParamsBlock = getListWithNamespaceOrEmpty(queryParams, "query params");


    String detail = String.format("The operation [%s:%s] has several parameters with the same 'parameterIdentifier:%s'. " +
        "Please check the following: %s%s%s",
                                  op.getHttpMethod(), op.getPath(),
                                  parameterIdentifier,
                                  uriParamsBlock,
                                  headersParamsBlock,
                                  queryParamsBlock);
    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }

  private String getListWithNamespaceOrEmpty(List<String> uriParams, String typeParams) {
    return uriParams.isEmpty() ? "" : String.format("\n\t" + typeParams + ": %s", String.join(", ", uriParams));
  }
}
