/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public abstract class AbstractResolverReferenceArgumentsExistsRule<T extends ResolverDefinitionDescriptor<T>>
    extends DescriptorValidationRule {

  private final String ERROR_TEMPLATE;

  public AbstractResolverReferenceArgumentsExistsRule(String resolverType) {
    super(format("When sending an argument to a %s it must exist in the resolver declaration", resolverType), EMPTY, ERROR);
    ERROR_TEMPLATE = new StringBuilder()
        .append("Argument '%s' does not exist in the '%s' ")
        .append(resolverType)
        .append(" declaration.")
        .toString();
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> validationResults = new ArrayList<>();

    for (ResolverReferenceDescriptor<T> reference : getResolverReferenceDescriptors(connectorDescriptor)) {
      final List<ResolverArgumentDescriptor> arguments = reference.getArguments();
      final List<ResolverParameterDescriptor> parameters = reference.getDeclaration().getParameters();

      final List<ValidationResult> errors = arguments.stream()
          .filter(x -> parameters.stream().noneMatch(y -> y.getName().equalsIgnoreCase(x.getName())))
          .map(x -> getValidationError(x, reference.getDeclaration()))
          .collect(toList());

      validationResults.addAll(errors);
    }

    return validationResults;
  }

  protected abstract List<ResolverReferenceDescriptor<T>> getResolverReferenceDescriptors(ConnectorDescriptor connectorDescriptor);

  private ValidationResult getValidationError(ArgumentDescriptor argumentDescriptor,
                                              ResolverDeclarationDescriptor<?> declaration) {
    return new ValidationResult(this,
                                format(ERROR_TEMPLATE, argumentDescriptor.getName(), declaration.getName()),
                                argumentDescriptor.getLocation());
  }
}
