/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.*;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.io.File;

import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.apache.commons.lang3.StringUtils.EMPTY;

public class FileSchemeExistRule extends DescriptorValidationRule {


  public FileSchemeExistRule() {
    super("Declared file scheme must exist in the file system",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();
    validationResults.addAll(verifyTriggersSchemes(connectorDescriptor.getTriggers()));
    validationResults.addAll(verifyEndPointsSchemes(connectorDescriptor.getEndpoints()));
    validationResults.addAll(verifyOperationAdapterSchemes(connectorDescriptor.getOperationAdapterDescriptors()));
    return validationResults;
  }

  private List<ValidationResult> verifyTriggersSchemes(List<TriggerDescriptor> triggerDescriptors) {
    List<ValidationResult> validationResults = new ArrayList<>();
    for (TriggerDescriptor triggerDescriptor : triggerDescriptors) {
      if (triggerDescriptor.getOutputTypeSchema() != null) {
        File file = triggerDescriptor.getOutputTypeSchema().getPath().toFile();
        ValidationResult validationResult =
            verifyPath(file, triggerDescriptor.getOutputTypeSchema(), triggerDescriptor.getLocation());
        if (validationResult != null) {
          validationResults.add(validationResult);
        }
      }
      validationResults.addAll(verifySchemesInTriggerParameters(triggerDescriptor.getParameters()));
    }
    return validationResults;
  }

  private List<ValidationResult> verifySchemesInTriggerParameters(List<TriggerParameterDescriptor> triggerParameterDescriptors) {
    List<ValidationResult> validationResults = new ArrayList<>();
    for (TriggerParameterDescriptor triggerParameterDescriptor : triggerParameterDescriptors) {
      if (triggerParameterDescriptor.getInputType() != null) {
        File file = triggerParameterDescriptor.getInputType().getPath().toFile();
        ValidationResult validationResult =
            verifyPath(file, triggerParameterDescriptor.getInputType(), triggerParameterDescriptor.getLocation());
        if (validationResult != null) {
          validationResults.add(validationResult);
        }
      }
    }
    return validationResults;
  }

  private List<ValidationResult> verifyEndPointsSchemes(List<EndPointDescriptor> endPointDescriptors) {
    List<ValidationResult> validationResults = new ArrayList<>();
    for (EndPointDescriptor endPointDescriptor : endPointDescriptors) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        if (operationDescriptor.getInputTypeSchema() != null) {
          File inputTypeSchema = operationDescriptor.getInputTypeSchema().getPath().toFile();
          ValidationResult validationResult =
              verifyPath(inputTypeSchema, operationDescriptor.getInputTypeSchema(), operationDescriptor.getLocation());
          if (validationResult != null) {
            validationResults.add(validationResult);
          }
        }
        if (operationDescriptor.getOutputTypeSchema() != null) {
          File outputTypeSchema = operationDescriptor.getOutputTypeSchema().getPath().toFile();
          ValidationResult validationResult =
              verifyPath(outputTypeSchema, operationDescriptor.getOutputTypeSchema(), operationDescriptor.getLocation());
          if (validationResult != null) {
            validationResults.add(validationResult);
          }
        }
      }
    }
    return validationResults;
  }

  private List<ValidationResult> verifyOperationAdapterSchemes(List<OperationAdapterDescriptor> operationAdapterDescriptors) {
    List<ValidationResult> validationResults = new ArrayList<>();
    for (OperationAdapterDescriptor operationAdapterDescriptor : operationAdapterDescriptors) {
      List<AuxiliarParameterBindingDescriptor> auxiliarParameterBindingDescriptors = new ArrayList<>();
      if (operationAdapterDescriptor.getRequestBindings() != null) {
        auxiliarParameterBindingDescriptors.addAll(operationAdapterDescriptor.getRequestBindings().getQueryParameters());
      }
      if (operationAdapterDescriptor.getRequestBindings() != null) {
        auxiliarParameterBindingDescriptors.addAll(operationAdapterDescriptor.getRequestBindings().getBodyBindings());
      }
      if (operationAdapterDescriptor.getRequestBindings() != null) {
        auxiliarParameterBindingDescriptors.addAll(operationAdapterDescriptor.getRequestBindings().getUriParameters());
      }
      for (AuxiliarParameterBindingDescriptor auxiliarParameterBindingDescriptor : auxiliarParameterBindingDescriptors) {
        if (auxiliarParameterBindingDescriptor.getInputType() != null) {
          File inputType = auxiliarParameterBindingDescriptor.getInputType().getPath().toFile();
          ValidationResult validationResult = verifyPath(inputType, auxiliarParameterBindingDescriptor.getInputType(),
                                                         auxiliarParameterBindingDescriptor.getLocation());
          if (validationResult != null) {
            validationResults.add(validationResult);
          }
        }
      }
      for (AuxiliarParameterDescriptor auxiliarParameterDescriptor : operationAdapterDescriptor.getParameters()) {
        if (auxiliarParameterDescriptor.getInputType() != null) {
          File inputType = auxiliarParameterDescriptor.getInputType().getPath().toFile();
          ValidationResult validationResult =
              verifyPath(inputType, auxiliarParameterDescriptor.getInputType(), auxiliarParameterDescriptor.getLocation());
          if (validationResult != null) {
            validationResults.add(validationResult);
          }
        }
      }
      if (operationAdapterDescriptor.getResponseBindings() != null
          && operationAdapterDescriptor.getResponseBindings().getBodyBinding().getTypeSchema() != null) {
        File mediaType = operationAdapterDescriptor.getResponseBindings().getBodyBinding().getTypeSchema().getPath().toFile();
        ValidationResult validationResult =
            verifyPath(mediaType, operationAdapterDescriptor.getResponseBindings().getBodyBinding().getTypeSchema(),
                       operationAdapterDescriptor.getResponseBindings().getLocation());
        if (validationResult != null) {
          validationResults.add(validationResult);
        }
      }
    }
    return validationResults;
  }

  private ValidationResult verifyPath(File file, SchemeDescriptor schemeDescriptor, DescriptorElementLocation location) {
    if (file.exists() && schemeDescriptor.getContent() == null) {
      return new ValidationResult(this,
                                  "the file with path"
                                      + schemeDescriptor.getPath().toAbsolutePath()
                                      + " cannot be accessed.",
                                  location);
    }
    if (!file.exists() && schemeDescriptor.getContent() == null) {
      return new ValidationResult(this, "The file with path: "
          + schemeDescriptor.getPath().toAbsolutePath() + " is not present in the file System.",
                                  location);
    }
    return null;
  }
}
