/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationAdapterDescriptors;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getOperationDescriptor;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.PaginationKind;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.ComparisonUtil;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;

public class PagedOperationsParameterMustExistRule extends DescriptorValidationRule {

  public PagedOperationsParameterMustExistRule() {
    super("Paged operations must contain a query parameter corresponding to the pagination parameter declared in the pagination definition",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();

    for (APIOperationModel apiOperation : apiModel.getOperationsModel()) {
      List<PaginationDeclarationDescriptor> paginationDeclarations = getPaginationDeclaration(connectorDescriptor, apiOperation);
      for (PaginationDeclarationDescriptor paginationDeclaration : paginationDeclarations) {
        if (!PaginationKind.HYPERMEDIA.getName().equalsIgnoreCase(paginationDeclaration.getKind())) {
          String parameterName = paginationDeclaration.getPaginationParameterName();
          if (apiOperation.getQueryParamsModel().stream().noneMatch(x -> ComparisonUtil.externalNameParamsComparison(
                                                                                                                     x.getExternalName(),
                                                                                                                     parameterName,
                                                                                                                     ParameterType
                                                                                                                         .valueOf(x
                                                                                                                             .getParameterType()
                                                                                                                             .name())))) {
            validationResults
                .add(getValidationError(paginationDeclaration, parameterName));
          }
        }

      }
    }

    return validationResults;
  }

  private List<PaginationDeclarationDescriptor> getPaginationDeclaration(ConnectorDescriptor connectorDescriptor,
                                                                         APIOperationModel apiOperationModel) {
    final OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperationModel);
    final List<OperationAdapterDescriptor> operationAdapterDescriptors =
        getOperationAdapterDescriptors(connectorDescriptor, apiOperationModel);

    List<PaginationDeclarationDescriptor> paginationDescriptors = new ArrayList<>();
    connectorDescriptor.getPaginations().stream()
        .filter(x -> isUsedInOperation(x.getName(), operationDescriptor)
            || isUsedInOperationAdapter(x.getName(), operationAdapterDescriptors))
        .forEach(paginationDescriptors::add);
    return paginationDescriptors;
  }

  private boolean isUsedInOperation(String name, OperationDescriptor operationDescriptor) {
    return operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())
        && name.equalsIgnoreCase(operationDescriptor.getPagination());
  }

  private boolean isUsedInOperationAdapter(String name, List<OperationAdapterDescriptor> operationAdapterDescriptors) {
    return operationAdapterDescriptors.stream()
        .anyMatch(operationAdapterDescriptor -> isNotBlank(operationAdapterDescriptor.getPagination())
            && name.equalsIgnoreCase(operationAdapterDescriptor.getPagination()));
  }


  private ValidationResult getValidationError(PaginationDeclarationDescriptor paginationDeclaration, String parameterName) {
    String description =
        "The pagination named '" + paginationDeclaration.getName() + "' declares a query parameter named '" + parameterName
            + "' that is not present in the API spec for the referencing operation";

    return new ValidationResult(this, description, paginationDeclaration.getLocation());
  }
}
