/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorSecurityKind;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.NOT_SUPPORTED;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.CUSTOM;

public class SecurityNotSupportedInApiMustHaveAnAssociatedCustomTypeInDescriptor extends DescriptorValidationRule {

  public SecurityNotSupportedInApiMustHaveAnAssociatedCustomTypeInDescriptor() {
    super("Unsupported security schemes need to be customized in the connector descriptor.",
          "Use 'custom' type in the connector descriptor to add a custom implementation.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationsResults = new ArrayList<>();
    List<SecuritySchemeBaseDescriptor> descriptorSecurities = connectorDescriptor.getSecurity();
    List<APISecuritySchemeModel> apiSecurities =
        getListOfApiSecurities(apiModel).stream().distinct().collect(Collectors.toList());
    List<APISecuritySchemeModel> apiSecuritiesNotSupportedWithNotMatchInDescriptor =
        descriptorContainsAtLeastOneAssociatedSecurity(apiModel, descriptorSecurities);
    if (!apiSecuritiesNotSupportedWithNotMatchInDescriptor.isEmpty()) {
      for (APISecuritySchemeModel apiSecuritySchemeModel : apiSecuritiesNotSupportedWithNotMatchInDescriptor) {
        validationsResults.add(getIsNotSupportedError(apiSecuritySchemeModel));
      }
    }
    for (APISecuritySchemeModel apiSecuritySchemeModel : apiSecurities)
      for (SecuritySchemeBaseDescriptor descriptorSecurity : descriptorSecurities) {
        if (hasAssociatedSecurityTypeInTheDescriptor(apiSecuritySchemeModel, descriptorSecurity)) {
          ValidationResult validationResult = getValidationError(apiSecuritySchemeModel, descriptorSecurity);
          if (validationsResults.stream().noneMatch(x -> x.getDetail().equals(validationResult.getDetail()))) {
            validationsResults.add(getValidationError(apiSecuritySchemeModel, descriptorSecurity));
          }
        }
      }
    return validationsResults;
  }

  private ValidationResult getValidationError(APISecuritySchemeModel apiSecuritySchemeModel,
                                              SecuritySchemeBaseDescriptor securitySchemeDescriptor) {
    String detail =
        "API Security scheme: "
            + "'" + apiSecuritySchemeModel.getName() + "'" +
            " was declared in the descriptor as type: "
            + "'" + securitySchemeDescriptor.getKind() + "'"
            + ". Only 'custom' is supported.";

    return new ValidationResult(this, detail, securitySchemeDescriptor.getLocation());
  }

  private ValidationResult getIsNotSupportedError(APISecuritySchemeModel apiSecuritySchemeModel) {
    String detail =
        "API Security scheme with name: "
            + apiSecuritySchemeModel.getName() +
            " is not supported nor customized in the connector descriptor.";

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }

  private boolean hasAssociatedSecurityTypeInTheDescriptor(APISecuritySchemeModel apiSecuritySchemeModel,
                                                           SecuritySchemeBaseDescriptor descriptorSecurity) {
    return (apiSecuritySchemeModel.getName().equals(descriptorSecurity.getName())
        && apiSecuritySchemeModel.getSecuritySchemeType().equals(NOT_SUPPORTED)
        && !descriptorSecurity.getKind().equals(DescriptorSecurityKind.CUSTOM)
        || apiSecuritySchemeModel.getSecuritySchemeType().equals(CUSTOM)
            && !descriptorSecurity.getKind().equals(DescriptorSecurityKind.CUSTOM));
  }

  private List<APISecuritySchemeModel> descriptorContainsAtLeastOneAssociatedSecurity(APIModel apiModel,
                                                                                      List<SecuritySchemeBaseDescriptor> descriptorSecurities) {
    List<APISecuritySchemeModel> returns = new ArrayList<>();
    List<APISecuritySchemeModel> apiSecuritySchemeModels = getListOfApiSecurities(apiModel);

    if (!apiSecuritySchemeModels.isEmpty() && descriptorSecurities.isEmpty()
        && !apiSecuritySchemeModels.stream().anyMatch(x -> !x.getSecuritySchemeType().equals(NOT_SUPPORTED))) {
      for (APISecuritySchemeModel apiSecuritySchemeModel : apiSecuritySchemeModels) {
        if (apiSecuritySchemeModel.getSecuritySchemeType().equals(NOT_SUPPORTED)) {
          returns.add(apiSecuritySchemeModel);
        }
      }
    }
    return returns;
  }

  private List<APISecuritySchemeModel> getListOfApiSecurities(APIModel apiModel) {
    List<APISecuritySchemeModel> apiSecuritySchemeModels = new ArrayList<>();
    for (APIOperationModel apiOperation : apiModel.getOperationsModel()) {
      for (APISecuritySchemeModel apiSecuritySchemeModel : apiOperation.getSecuritySchemesModel()) {
        if (apiSecuritySchemeModels.stream().noneMatch(x -> x.getName().equals(apiSecuritySchemeModel.getName()))) {
          apiSecuritySchemeModels.add(apiSecuritySchemeModel);
        }
      }
    }
    return apiSecuritySchemeModels;
  }
}
