/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getSecSchemesSupportedByApi;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

public class SecuritySchemeMustBeSupportedByTheApiRule extends DescriptorValidationRule {

  public SecuritySchemeMustBeSupportedByTheApiRule() {
    super("All security schemes used in the descriptor must be supported by the API specification.",
          EMPTY,
          ERROR);
  }


  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> validationsResults = new ArrayList<>();

    Set<String> secSchemesSupportedByApi = getSecSchemesSupportedByApi(apiModel);

    List<Pair<String, DescriptorElementLocation>> securitySchemesAndLocations = connectorDescriptor.getSecurity().stream()
        .map(x -> Pair.of(x.getName(), x.getLocation()))
        .distinct()
        .collect(Collectors.toList());

    for (Pair<String, DescriptorElementLocation> scheme : securitySchemesAndLocations) {
      if (!secSchemesSupportedByApi.contains(scheme.getLeft())) {
        validationsResults
            .add(getValidationError(scheme, secSchemesSupportedByApi));
      }
    }

    return validationsResults;
  }

  private ValidationResult getValidationError(Pair<String, DescriptorElementLocation> unsupportedScheme,
                                              Set<String> securitySchemesSupportedByApi) {
    String detail =
        "The Security Scheme named: '"
            + unsupportedScheme.getLeft() +
            "' is not present in the API specification. The supported security schemes are: '"
            + String.join("', '", securitySchemesSupportedByApi)
            + "'.";

    return new ValidationResult(this, detail, unsupportedScheme.getRight());
  }


}
