/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.NOT_SUPPORTED;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;

import java.util.ArrayList;
import java.util.List;

public class AtLeastOneSupportedSecuritySchemeRule extends ConnectorModelValidationRule {

  // R004
  public AtLeastOneSupportedSecuritySchemeRule() {
    super("The API spec defines a security scheme that is not supported.",
          "Unsupported security schemes won't be generated in the connector.",
          WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    return apiModel.getOperationsModel().stream()
        .filter(opm -> ValidationUtils.apiOperationIsPresentInConnectorModel(opm, connectorModel))
        .filter(opm -> allSecuritySchemesUnsupported(opm, connectorModel))
        .map(this::getValidationError)
        .collect(toList());
  }

  private boolean allSecuritySchemesUnsupported(APIOperationModel apiOperationModel, ConnectorModel connectorModel) {
    List<ConnectorSecurityScheme> connectorSecuritySchemes = new ArrayList<>();
    for (ConnectorOperation connectorOperation : connectorModel.getOperations())
      for (ConnectorSecurityScheme connectorSecurityScheme : connectorOperation.getSecuritySchemes()) {
        if (!connectorSecuritySchemes.stream().anyMatch(x -> x.getName().equals(connectorSecurityScheme.getName()))) {
          connectorSecuritySchemes.add(connectorSecurityScheme);
        }
      }

    for (APISecuritySchemeModel apiSecuritySchemeModel : apiOperationModel.getSecuritySchemesModel())
      if (connectorSecuritySchemes.stream().anyMatch(x -> x.getName().equals(apiSecuritySchemeModel.getName()))) {
        return false;
      }

    return !apiOperationModel.getSecuritySchemesModel().isEmpty() &&
        apiOperationModel.getSecuritySchemesModel().stream().allMatch(x -> x.getSecuritySchemeType().equals(NOT_SUPPORTED));
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String detail =
        "API Operation with PATH: "
            + apiOperationModel.getPath()
            + " and METHOD: "
            + apiOperationModel.getHttpMethod()
            + " will be generated without securityScheme since an unsupported type was declared.";

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
