/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiOperation;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getApiParameter;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.ArrayList;
import java.util.List;

public class OperationAdapterRequiredParametersMustNotBeIgnoredRule extends DescriptorValidationRule {

  public OperationAdapterRequiredParametersMustNotBeIgnoredRule() {
    super("Required parameter bindings must not be ignored.", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    List<OperationAdapterDescriptor> operationAdapterDescriptors = connectorDescriptor.getOperationAdapterDescriptors();
    if (!operationAdapterDescriptors.isEmpty()) {
      for (OperationAdapterDescriptor operationAdapterDescriptor : operationAdapterDescriptors) {
        if (operationAdapterDescriptor.getRequestBindings() != null) {
          analyzeParametersBinding(results, operationAdapterDescriptor,
                                   operationAdapterDescriptor.getRequestBindings().getUriParameters(), URI, apiModel);
          analyzeParametersBinding(results, operationAdapterDescriptor,
                                   operationAdapterDescriptor.getRequestBindings().getQueryParameters(), QUERY, apiModel);
          analyzeParametersBinding(results, operationAdapterDescriptor,
                                   operationAdapterDescriptor.getRequestBindings().getHeaders(), HEADER, apiModel);
        }
      }
    }

    return results;
  }

  private void analyzeParametersBinding(List<ValidationResult> results, OperationAdapterDescriptor operationAdapterDescriptor,
                                        List<AuxiliarParameterBindingDescriptor> parameterBindingDescriptors,
                                        ParameterType parameterType, APIModel apiModel) {

    APIOperationModel apiOperationModel = getApiOperation(apiModel, operationAdapterDescriptor);
    for (AuxiliarParameterBindingDescriptor binding : parameterBindingDescriptors) {
      if (binding.getIgnored() != null && binding.getIgnored()) {
        APIParameterModel apiParameter = getApiParameter(apiOperationModel, parameterType, binding.getName());
        if (apiParameter != null && (URI.equals(parameterType) || apiParameter.isRequired())) {
          results.add(getValidationError(operationAdapterDescriptor, binding, parameterType));
        }
      }
    }

  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              AuxiliarParameterBindingDescriptor binding,
                                              ParameterType parameterType) {
    String detail = parameterType
        + " binding '"
        + binding.getName()
        + "' defined in Operation with name: "
        + operationAdapterDescriptor.getOperationId()
        + " cannot be ignored as it's required.";

    return new ValidationResult(this, detail, operationAdapterDescriptor.getLocation());
  }

}
