/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.ATTRIBUTES_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.LINK;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.WeaveExpressionDescriptor;
import org.mule.weave.v2.parser.phase.CompilationException;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class PagedOperationsExpressionMustCompileRule extends DescriptorValidationRule {

  private static final String[] IMPLICIT_INPUTS = {PAYLOAD_VAR, ATTRIBUTES_VAR};
  private static final String[] IMPLICIT_INPUTS_HYPERMEDIA = {PAYLOAD_VAR, ATTRIBUTES_VAR, LINK};

  public PagedOperationsExpressionMustCompileRule() {
    super("Expression is invalid",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();

    for (PaginationDeclarationDescriptor paginationDeclaration : connectorDescriptor.getPaginations()) {
      if (paginationDeclaration != null) {
        if (paginationDeclaration.getNextTokenExpression() != null) {
          validateExpression(connectorDescriptor, paginationDeclaration,
                             paginationDeclaration.getNextTokenExpression(), IMPLICIT_INPUTS, "nextToken",
                             validationResults);
        }
        if (paginationDeclaration.getPagingResponseExpression() != null) {
          validateExpression(connectorDescriptor, paginationDeclaration,
                             paginationDeclaration.getPagingResponseExpression(),
                             IMPLICIT_INPUTS, "pagingResponse",
                             validationResults);
        }
        if (paginationDeclaration.getPageCountExpression() != null) {
          validateExpression(connectorDescriptor, paginationDeclaration,
                             paginationDeclaration.getPageCountExpression(), IMPLICIT_INPUTS, "pageCount",
                             validationResults);
        }
        if (paginationDeclaration.getNextUrlExpression() != null) {
          validateExpression(connectorDescriptor, paginationDeclaration,
                             paginationDeclaration.getNextUrlExpression(),
                             IMPLICIT_INPUTS_HYPERMEDIA, "nextUrl",
                             validationResults);
        }
      }
    }

    return validationResults;
  }

  private void validateExpression(ConnectorDescriptor connectorDescriptor,
                                  PaginationDeclarationDescriptor paginationDeclarationDescriptor,
                                  WeaveExpressionDescriptor expression, String[] implicitInputs, String expressionName,
                                  List<ValidationResult> validationResults) {
    try {
      ExpressionHandlerUtils.compileDataWeaveScript(expression.getExpression(), implicitInputs);
    } catch (CompilationException e) {
      validationResults.add(getValidationError(connectorDescriptor, paginationDeclarationDescriptor,
                                               expressionName, expression.getExpression(), e));
    }
  }

  private ValidationResult getValidationError(ConnectorDescriptor connectorDescriptor,
                                              PaginationDeclarationDescriptor paginationDeclarationDescriptor,
                                              String expressionName, String expression, CompilationException e) {
    String detail = format("'%s' expression '%s' in %s is not valid, error: \n%s", expressionName, expression,
                           paginationDeclarationDescriptor.getName(), e.getMessage());
    return new ValidationResult(this, detail, paginationDeclarationDescriptor.getLocation());
  }
}
