/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.PaginationKind;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterRequestBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.RequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class PagedParametersMustNotBeIgnoredRule extends DescriptorValidationRule {

  public PagedParametersMustNotBeIgnoredRule() {
    super("for pageNumberPagination the queryParameter page_number can be ignored ",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();

    validationResults.addAll(validateOperations(connectorDescriptor));
    validationResults.addAll(validateOperationAdapters(connectorDescriptor));

    return validationResults;
  }

  private List<ValidationResult> validateOperations(ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();
    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        PaginationDeclarationDescriptor paginationDeclaration =
            getPaginationDeclaration(operationDescriptor.getPagination(), connectorDescriptor);
        if (paginationDeclaration != null
            && PaginationKind.PAGE_NUMBER.getName().equalsIgnoreCase(paginationDeclaration.getKind())) {
          RequestDescriptor requestDescriptor = operationDescriptor.getExpects();
          if (requestDescriptor != null) {
            List<ParameterDescriptor> parameters = requestDescriptor.getQueryParameter();
            for (ParameterDescriptor parameter : parameters) {
              if (parameter.getParamName().equals(paginationDeclaration.getPaginationParameterName())
                  && parameter.isIgnored() != null && parameter.isIgnored()) {
                validationResults
                    .add(getValidationError(format("%s-%s", operationDescriptor.getMethod(), endPointDescriptor.getPath()),
                                            parameter.getLocation(),
                                            paginationDeclaration.getName(), parameter.getParamName(),
                                            paginationDeclaration.getKind(), false));
              }
            }
          }
        }
      }
    }
    return validationResults;
  }

  private List<ValidationResult> validateOperationAdapters(ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();
    for (OperationAdapterDescriptor operationAdapterDescriptor : connectorDescriptor.getOperationAdapterDescriptors()) {
      PaginationDeclarationDescriptor paginationDeclaration =
          getPaginationDeclaration(operationAdapterDescriptor.getPagination(), connectorDescriptor);
      if (paginationDeclaration != null
          && PaginationKind.PAGE_NUMBER.getName().equalsIgnoreCase(paginationDeclaration.getKind())) {
        AuxiliarParameterRequestBindingsDescriptor requestBindings = operationAdapterDescriptor.getRequestBindings();
        if (requestBindings != null) {
          List<AuxiliarParameterBindingDescriptor> queryParameters = requestBindings.getQueryParameters();
          for (AuxiliarParameterBindingDescriptor parameter : queryParameters) {
            if (parameter.getName().equals(paginationDeclaration.getPaginationParameterName())
                && parameter.getIgnored() != null && parameter.getIgnored()) {
              validationResults
                  .add(getValidationError(operationAdapterDescriptor.getOperationId(),
                                          parameter.getLocation(),
                                          paginationDeclaration.getName(), parameter.getName(),
                                          paginationDeclaration.getKind(), true));
            }
          }
        }
      }
    }
    return validationResults;
  }

  private PaginationDeclarationDescriptor getPaginationDeclaration(String paginationName,
                                                                   ConnectorDescriptor connectorDescriptor) {
    PaginationDeclarationDescriptor paginationDescriptor = null;
    if (isNotBlank(paginationName)) {
      paginationDescriptor = connectorDescriptor.getPaginations().stream()
          .filter(x -> x.getName().equalsIgnoreCase(paginationName))
          .findFirst().orElse(null);
    }

    return paginationDescriptor;
  }

  private ValidationResult getValidationError(String operationIdentifier, DescriptorElementLocation paramenterLocation,
                                              String namePagination, String parameterName, String typePagination,
                                              boolean isOperationAdapter) {
    String detail =
        (isOperationAdapter ? "Custom Operation : " : "Endpoint operation : ")
            + operationIdentifier +
            " use Pagination "
            + namePagination
            + " with a parameter with name "
            + parameterName + " which was ignored and is required for type pagination "
            + typePagination +
            ".";

    return new ValidationResult(this, detail, paramenterLocation);
  }
}
