/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class BodyIdentifierScriptMustNotRepeatRule extends ConnectorModelValidationRule {

  public BodyIdentifierScriptMustNotRepeatRule() {
    super("Body identifiers must not be repeated",
          "There are operations with the same 'bodyIdentifier'",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    Map<String, List<ConnectorOperation>> operationIdentifiers = new HashMap<>();

    List<ConnectorOperation> operationsWithBody = connectorModel.getOperations().stream()
        .filter(operation -> operation.getInputMetadata() != null
            && operation.getBody() != null
            && !operation.isAdapter())
        .collect(Collectors.toList());
    for (ConnectorOperation op : operationsWithBody) {
      if (!operationIdentifiers.containsKey(op.getBody().getBodyIdentifier())) {
        operationIdentifiers.put(op.getBody().getBodyIdentifier(), new ArrayList<>());
      }
      operationIdentifiers.get(op.getBody().getBodyIdentifier()).add(op);
    }

    List<ValidationResult> errors = operationIdentifiers.entrySet().stream()
        .filter(entry -> entry.getValue().size() > 1)
        .map(entry -> getValidationErrors(entry.getKey(), entry.getValue()))
        .flatMap(Collection::stream)
        .collect(Collectors.toList());

    return errors;
  }

  private List<ValidationResult> getValidationErrors(String collideBodyIdentifier, List<ConnectorOperation> ops) {
    List<ValidationResult> partialErrors = ops.stream().map(op -> {
      String detail;
      if (op.isAdapter()) {
        detail = format("The operation [%s] body identifier is already defined [%s].", op.getOperationIdentifier(),
                        collideBodyIdentifier);
      } else {
        String operationName = op.getHttpMethod() + ":" + op.getPath();
        detail = format("The operation [%s] has a 'bodyIdentifier: %s' value that's already defined.", operationName,
                        collideBodyIdentifier);
      }

      DescriptorElementLocation loc = DescriptorElementLocation.builder().empty();
      return new ValidationResult(this, detail, loc);
    })
        .collect(Collectors.toList());
    return partialErrors;
  }
}
