/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor.connectormodelbuilder;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;
import java.util.stream.Collectors;

public class BaseTriggerMustExistRule extends DescriptorConnectorModelBuilderValidationRule {

  public BaseTriggerMustExistRule() {
    super("Base Trigger must exist", EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor,
                                         ConnectorModelBuilder connectorModelBuilder) {

    return connectorDescriptor.getTriggers().stream()
        .filter(triggerDescriptor -> triggerDescriptor.getBase() != null)
        .filter(triggerDescriptor -> !baseTriggerExist(triggerDescriptor, connectorDescriptor)
            && !baseTriggerExist(triggerDescriptor, connectorModelBuilder))
        .map(this::getValidationError)
        .collect(Collectors.toList());
  }

  private boolean baseTriggerExist(TriggerDescriptor triggerDescriptor, ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getTriggers().stream()
        .anyMatch(trigger -> triggerDescriptor.getBase().equals(trigger.getName()));
  }

  private boolean baseTriggerExist(TriggerDescriptor triggerDescriptor, ConnectorModelBuilder connectorModelBuilder) {
    return connectorModelBuilder.getTriggerBuildersByName(triggerDescriptor.getBase()).isPresent();
  }

  private ValidationResult getValidationError(TriggerDescriptor triggerDescriptor) {
    String detail =
        "Trigger with name: '"
            + triggerDescriptor.getName()
            + "' defines a base trigger: '"
            + triggerDescriptor.getBase()
            + "' that does not exist";

    return new ValidationResult(this, detail, triggerDescriptor.getLocation());
  }
}
