/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.PaginationType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.*;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.*;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class PagedParametersMustNotBeIgnoredRule extends DescriptorValidationRule {

  public PagedParametersMustNotBeIgnoredRule() {
    super("for pageNumberPagination the queryParameter page_number can be ignored ",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();

    for (APIOperationModel apiOperation : apiModel.getOperationsModel()) {
      OperationDescriptor operationDescriptor = getOperationDescriptor(connectorDescriptor, apiOperation);
      PaginationDeclarationDescriptor paginationDeclaration = getPaginationDeclaration(operationDescriptor, connectorDescriptor);
      if (paginationDeclaration != null
          && PaginationType.PAGE_NUMBER.getName().equalsIgnoreCase(paginationDeclaration.getType())) {
        RequestDescriptor requestDescriptor = operationDescriptor.getExpects();
        if (requestDescriptor != null) {
          List<ParameterDescriptor> parameters = requestDescriptor.getQueryParameter();
          for (ParameterDescriptor parameter : parameters) {
            if (parameter.getParamName().equals(paginationDeclaration.getPaginationParameterName())
                && parameter.isIgnored() != null && parameter.isIgnored()) {
              validationResults
                  .add(getValidationError(apiOperation, parameter,
                                          paginationDeclaration.getName(), parameter.getParamName(),
                                          paginationDeclaration.getType()));
            }
          }
        }

      }
    }
    return validationResults;
  }

  private PaginationDeclarationDescriptor getPaginationDeclaration(OperationDescriptor operationDescriptor,
                                                                   ConnectorDescriptor connectorDescriptor) {
    PaginationDeclarationDescriptor paginationDescriptor = null;
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())) {
      paginationDescriptor = connectorDescriptor.getPaginations().stream()
          .filter(x -> x.getName().equalsIgnoreCase(operationDescriptor.getPagination()))
          .findFirst().orElse(null);
    }

    return paginationDescriptor;
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel, ParameterDescriptor paramDescriptor,
                                              String namePagination, String parameterName, String typePagination) {
    String detail =
        "API Operation with PATH: "
            + apiOperationModel.getPath() +
            " and METHOD: "
            + apiOperationModel.getHttpMethod().toUpperCase() +
            " use Pagination "
            + namePagination
            + " with a parameter with name "
            + parameterName + " which was ignored and is required for type pagination "
            + typePagination +
            ".";

    return new ValidationResult(this, detail, paramDescriptor.getLocation());
  }
}
