/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil.isFriendlyName;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.List;

public class OperationNameMustBeFriendlyRule extends ConnectorModelValidationRule {

  // R012
  public OperationNameMustBeFriendlyRule() {
    super("Operation display name seems not to be user friendly.",
          "The display name should be a human readable name separated with spaces.",
          WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    return connectorModel.getOperations().stream()
        .filter(x -> !isFriendlyName(x.getDisplayName()))
        .map(this::getValidationError)
        .collect(toList());
  }

  private ValidationResult getValidationError(ConnectorOperation connectorOperation) {
    String detail = format("Operation with PATH: %s and METHOD: %s display name is '%s' and it is not friendly.",
                           connectorOperation.getPath(),
                           connectorOperation.getHttpMethod().name(),
                           connectorOperation.getDisplayName());

    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }
}
