/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.NOT_SUPPORTED;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;

import java.util.ArrayList;
import java.util.List;

public class NotSupportedSecuritySchemeRule extends ConnectorModelValidationRule {

  // R010
  public NotSupportedSecuritySchemeRule() {
    super("The API spec defines a security scheme that is not supported",
          EMPTY,
          WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    return apiModel.getOperationsModel().stream()
        .filter(opm -> ValidationUtils.apiOperationIsPresentInConnectorModel(opm, connectorModel))
        .flatMap(apiOp -> apiOp.getSecuritySchemesModel().stream())
        .filter(securityScheme -> securityScheme.getSecuritySchemeType().equals(NOT_SUPPORTED))
        .filter(opm -> notExistAssociatedCustomSecurityInDescriptor(apiModel.getOperationsModel(), connectorModel))
        .map(APISecuritySchemeModel::getName)
        .distinct()
        .map(this::getValidationError)
        .collect(toList());
  }

  private ValidationResult getValidationError(String securitySchemeName) {
    final String detail = "Security scheme '" + securitySchemeName + "' type is not supported";
    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }

  private boolean notExistAssociatedCustomSecurityInDescriptor(List<APIOperationModel> apiOperationModel,
                                                               ConnectorModel connectorModel) {

    List<ConnectorSecurityScheme> connectorSecuritySchemes = new ArrayList<>();
    for (ConnectorOperation connectorOperation : connectorModel.getOperations())
      for (ConnectorSecurityScheme connectorSecurityScheme : connectorOperation.getSecuritySchemes()) {
        if (!connectorSecuritySchemes.stream().anyMatch(x -> x.getName().equals(connectorSecurityScheme.getName()))) {
          connectorSecuritySchemes.add(connectorSecurityScheme);
        }
      }

    for (APIOperationModel asd : apiOperationModel) {
      for (APISecuritySchemeModel apiSecuritySchemeModel : asd.getSecuritySchemesModel())
        if (connectorSecuritySchemes.stream().anyMatch(x -> x.getName().equals(apiSecuritySchemeModel.getName()))) {
          return false;
        }
    }
    return true;
  }
}
