/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.BaseParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.MultipartTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class ParameterInternalNameMustNotRepeatRule extends ConnectorModelValidationRule {

  public ParameterInternalNameMustNotRepeatRule() {
    super("Parameter's internalName must not be repeated among operation.",
          "The internal name in some parameters, used for java classes generation and generated from 'parameterIdentifier' script, is colliding. "
              +
              "Please re-arrange the 'parameterIdentifier' script accordingly to avoid collisions",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    return connectorModel.getOperations()
        .stream()
        .map(this::validateParametersIdentifiers)
        .flatMap(List::stream)
        .collect(Collectors.toList());
  }

  private List<ValidationResult> validateParametersIdentifiers(ConnectorOperation operation) {
    Map<String, List<BaseParameter>> parametersInternalName = new HashMap<>();

    Consumer<BaseParameter> collisionSearcher = parameter -> {
      if (!parametersInternalName.containsKey(parameter.getInternalName())) {
        parametersInternalName.put(parameter.getInternalName(), new ArrayList<>());
      }
      parametersInternalName.get(parameter.getInternalName()).add(parameter);
    };
    operation.getUriParameters().forEach(collisionSearcher);
    operation.getHeaders().forEach(collisionSearcher);
    operation.getQueryParameters().forEach(collisionSearcher);
    operation.getParameters().ifPresent(parameters -> parameters.forEach(collisionSearcher));

    // Checking collisions in security schemas of the operation
    for (ConnectorSecurityScheme sc : operation.getSecuritySchemes()) {
      sc.getQueryParameters().forEach(collisionSearcher);
    }

    if (operation.getBody() != null) {
      // TODO RSDK-620: [Tech Debt] Refactor parameter hierarchy, body should be thought as another param in the operation
      ParameterBuilder bodyParamBuilder =
          new ParameterBuilder(ParameterType.BODY, "body", operation.getBody().getBodyIdentifier());
      Parameter bodyParam = bodyParamBuilder.buildParameter(new TypeSchemaPool(), Optional.empty());
      collisionSearcher.accept(bodyParam);
    }
    if (operation.getInputMetadata() instanceof MultipartTypeDefinition) {
      // TODO RSDK-620: [Tech Debt] Refactor parameter hierarchy, multipart should be thought as another param in the operation,
      // within a body probably
      MultipartTypeDefinition multipartContent = (MultipartTypeDefinition) operation.getInputMetadata();
      multipartContent.getParts().forEach(collisionSearcher);
    }

    List<ValidationResult> validationResults = parametersInternalName.entrySet().stream()
        .filter(entry -> entry.getValue().size() > 1)
        .map(entry -> getValidationError(entry.getKey(), operation, entry.getValue()))
        .collect(Collectors.toList());
    return validationResults;
  }

  private ValidationResult getValidationError(String parameterIdentifier, ConnectorOperation op, List<BaseParameter> parameters) {
    List<String> uriParams = new ArrayList<>();
    List<String> headerParams = new ArrayList<>();
    List<String> queryParams = new ArrayList<>();
    List<String> bodyParams = new ArrayList<>();
    List<String> multipartParams = new ArrayList<>();
    List<String> auxiliarParams = new ArrayList<>();

    parameters.forEach(p -> {
      switch (p.getParameterType()) {
        case URI:
          uriParams.add(p.getExternalName());
          break;
        case QUERY:
          queryParams.add(p.getExternalName());
          break;
        case HEADER:
          headerParams.add(p.getExternalName());
          break;
        case BODY:
          bodyParams.add(p.getExternalName());
          break;
        case PART:
          multipartParams.add(p.getExternalName());
          break;
        case AUXILIAR:
          auxiliarParams.add(p.getExternalName());
        default:
          break;
      }
    });

    String uriParamsBlock = getListWithNamespaceOrEmpty(uriParams, "URI params");
    String headersParamsBlock = getListWithNamespaceOrEmpty(headerParams, "headers");
    String queryParamsBlock = getListWithNamespaceOrEmpty(queryParams, "query params");
    String bodyBlock = getListWithNamespaceOrEmpty(bodyParams, "Body param");
    String multipartParamsBlock = getListWithNamespaceOrEmpty(multipartParams, "multipart params");
    String auxiliarParamsBlock = getListWithNamespaceOrEmpty(auxiliarParams, "auxiliar params");

    String detail = String.format("The operation [%s:%s] has several parameters with the same internal name [%s]'. " +
        "Please check the following: %s%s%s%s%s%s",
                                  op.getHttpMethod(), op.getPath(),
                                  parameterIdentifier,
                                  uriParamsBlock,
                                  headersParamsBlock,
                                  queryParamsBlock,
                                  bodyBlock, multipartParamsBlock, auxiliarParamsBlock);
    return new ValidationResult(this, detail, DescriptorElementLocation.builder().empty());
  }

  private String getListWithNamespaceOrEmpty(List<String> uriParams, String typeParams) {
    return uriParams.isEmpty() ? "" : String.format("\n\t" + typeParams + ": %s", String.join(", ", uriParams));
  }
}
